




def boundary_callback(u, v, data):
    GinvT = data.GinvT()
    n     = data.n()

    Du = grad(u, GinvT)
    return inner(inner(n, Du), v)


# example low level forms:

def mass_v0(itg):
    """Compact version"""
    for i in range(itg.num_v_dofs(0)):
        for j in range(itg.num_v_dofs(1)):
            itg.A[(i,j)] = inner( itg.v_basis(0, i), itg.v_basis(1, j) )

def mass_v1(itg):
    """More elaborate version"""
    for i in range(itg.num_v_dofs(0)):
        for j in range(itg.num_v_dofs(1)):
            u = itg.v_basis(0, i)
            v = itg.v_basis(1, j)
            itg.A[(i,j)] = inner(u, v)

def mass_v2(itg):
    """'Jacobi' version"""
    dofs = itg.w_dofs(0)
    usum = itg.w_sum(0)
    for i in range(itg.num_v_dofs(0)):
        for j in range(itg.num_v_dofs(1)):
            u = diff(usum, dofs[i])
            v = itg.v_basis(1, j)
            itg.A[(i,j)] = inner(u, v)

def stiffness_v1(itg):
    """symbolic-grad-version"""
    GinvT = itg.GinvT()

    for i in range(itg.num_v_dofs(0)):
        for j in range(itg.num_v_dofs(1)):
            u  = itg.v_basis(0, i)
            v  = itg.v_basis(1, j)

            Du = grad(u, GinvT)
            Dv = grad(v, GinvT)

            #Du = itg.add_token(Du) # TODO: do it like this?
            #Dv = itg.add_token(Dv)

            itg.A[(i,j)] = inner(Du, Dv)

def stiffness_v2(itg):
    """'Manual Jacobi' version"""
    GinvT = itg.GinvT()

    dofs = itg.w_dofs(0)
    usum = itg.w_sum(0)

    Du = grad(usum, GinvT)

    for i in range(itg.num_v_dofs(0)):
        for j in range(itg.num_v_dofs(1)):
            v  = itg.v_basis(1, j)
            Dv = grad(v, GinvT)

            integrand = inner(Du, Dv)
            itg.A[(i,j)] = diff(integrand, dofs[i])



# example prototype forms for alternate syntax:

def stiffness_short(itg):
    """alternative short version"""
    print "Non-working prototype!"
    #f, g = itg.w_sums(0, 1) # TODO: do it like this?
    for (i, j) in itg.indices:
        u, v    = itg.v_basis_functions(i, j) # TODO: do it like this?
        Du, Dv  = itg.v_basis_function_gradients(i, j) # TODO: do it like this?
        itg.A[(i,j)] = inner(Du, Dv)

def stiffness_v0(itg):
    """grad-from-itg-version"""
    print "Non-working prototype!"
    for i in range(itg.num_v_dofs(0)):
        for j in range(itg.num_v_dofs(1)):
            u  = itg.v_basis(0, i)
            v  = itg.v_basis(1, j)

            Du = itg.grad_v(0, i) # TODO: do it like this?
            Dv = itg.grad_v(1, j)

            itg.A[(i,j)] = inner(Du, Dv)


# example nonlinear forms for automatic Jacobi creation:

def nonlinear_boundary_F(itg):
    """F_i(w,f) = \int_\dOmega f(x) e^{t \cdot u(x)} (n \cdot grad u) v_i(x) ds
       Coefficients:
          w (u from last iteration)
          f
    """
    GinvT = itg.GinvT()
    n     = itg.n()
    t     = itg.t()

    w = itg.w_sum(0)
    f = itg.w_sum(1)

    Dw  = grad(w, GinvT)
    nDw = inner(n, Dw)
    wn  = inner(n, w)
    wt  = inner(t, w)
    exp_wt = exp( wt )

    assert isinstance(n, matrix)
    assert isinstance(w, matrix)
    assert n.nops() == w.nops()
    assert not isinstance(wn.evalm(), matrix)
    assert not isinstance(wt.evalm(), matrix)

    tmp = f * exp_wt * nDw

    for i in range(itg.num_v_dofs(0)):
        v = itg.v_basis(0, i)
        itg.A[(i,)] = inner(tmp, v)


def nonlinear_F(itg):
    """Nonlinear test form F_i(w) = \int_\Omega w^2(x) w(x) v_i(x) dx]"""
    w = itg.w_sum(0)
    w2 = inner(w, w)
    for i in range(itg.num_v_dofs(0)):
        v = itg.v_basis(0, i)
        itg.A[(i,)] = w2 * inner(w, v)



def stiffness_with_tokens_matrix(u, v, M, data):
    GinvT = data.GinvT()

    Du = grad(u, GinvT)
    Dv = grad(v, GinvT)

    # create manual temporary variables:
    Du = data.add_token("Du", Du)
    Dv = data.add_token("Dv", Dv)

    return inner(M * Du, Dv)



if __name__ == "__main__":

    print_forms = False
    print_forms = True

    nsd = 2
    SyFi.initSyFi(nsd)
    polygon = SyFi.ReferenceTriangle()
    fe0 = SyFi.P0(polygon)
    fe1 = SyFi.Lagrange(polygon, 1)
    fe2 = SyFi.Lagrange(polygon, 2)
    vfe0 = SyFi.VectorP0(polygon)
    vfe1 = SyFi.VectorLagrange(polygon, 1)
    vfe2 = SyFi.VectorLagrange(polygon, 2)
    tfe0 = SyFi.TensorP0(polygon)
    tfe1 = SyFi.TensorLagrange(polygon, 1)
    tfe2 = SyFi.TensorLagrange(polygon, 2)


    fe_list = [fe1, fe1]
    #form = UserForm(rank=2, num_coefficients=0, name="mass", fe_list=fe_list, symbolic=False, quad_order=3)
    form = UserForm(rank=2, num_coefficients=0, name="mass", fe_list=fe_list, symbolic=True, quad_order=-1)
    if print_forms: print form
    form.sanity_check()

    citg = form.cell_integral()
    mass_v0(citg)
    if print_forms: print form
    form.sanity_check()


    # testing callback version of user interface:

    fe_list = [fe1, fe1]
    form = CallbackForm(name="mass", rank=2, num_coefficients=0, fe_list=fe_list, symbolic=True, quad_order=-1,
                      cell_integrands=[mass_callback])
    if print_forms: print form
    form.sanity_check()


    fe_list = [vfe1, vfe1, fe0]
    mf = CallbackForm(name="mass_with_c", rank=2, num_coefficients=1, fe_list=fe_list, symbolic=True, quad_order=-1,
                      cell_integrands=[mass_with_c_callback])
    if print_forms: print mf

    fe_list = [vfe1, vfe1, tfe0]
    form = CallbackForm(rank=2, fe_list=fe_list, symbolic=True, quad_order=-1,
                      cell_integrands=[stiffness_with_M_callback])
    if print_forms: print form
    form.sanity_check()

    fe_list = [fe2, fe2]
    form = CallbackForm(rank=2, fe_list=fe_list, symbolic=True, quad_order=-1,
                      exterior_facet_integrands=[boundary_callback])
    if print_forms: print form
    form.sanity_check()


    # testing coefficient counting
    fe_list = [fe1, fe0, fe2]
    form = UserForm(rank=0, num_coefficients=3, fe_list=fe_list)
    itg = form.cell_integral()
    if print_forms: print form
    form.sanity_check()


    # testing Jacobi version of user interface:
    fe_list = [vfe1, vfe1, fe0]
    F = UserForm(rank=1, num_coefficients=2, name="F", fe_list=fe_list, symbolic=True, quad_order=-1)

    itg = F.cell_integral()
    nonlinear_F(itg)
    itg.sanity_check()

    itg = F.exterior_facet_integral()
    nonlinear_boundary_F(itg)
    if print_forms: print F
    itg.sanity_check()

    J = Jacobi(F)
    if print_forms: print J
    J.sanity_check()

