% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table.R
\name{weighted_table}
\alias{weighted_table}
\title{Weighted table}
\usage{
weighted_table(..., weights, na_remove = FALSE)
}
\arguments{
\item{...}{Factors of equal length to use in the weighted table. If the
\code{...} are named, those names will propagate onto the "dimnames names" of
the resulting table. At least one factor must be provided.}

\item{weights}{A double vector of weights used to fill the cells of the
weighted table. This must be the same length as the factors provided in
\code{...}.}

\item{na_remove}{A single \code{TRUE} or \code{FALSE} for handling whether or not
missing values in \code{weights} should be removed when summing up the weights.}
}
\value{
The weighted table as an array of double values.
}
\description{
\code{weighted_table()} computes a weighted contingency table based on factors
provided in \code{...} and a double vector of weights provided in \code{weights}. It
can be seen as a weighted extension to \code{\link[base:table]{base::table()}} and an alternative
to \code{\link[stats:xtabs]{stats::xtabs()}}.

\code{weighted_table()} always uses the \emph{exact} set of levels returned by
\code{levels()} when constructing the table. This results in the following
properties:
\itemize{
\item Missing values found in the factors are never included in the table unless
there is an explicit \code{NA} factor level. If needed, this can be added to a
factor with \code{\link[base:factor]{base::addNA()}} or \code{forcats::fct_expand(x, NA)}.
\item Levels found in the factors that aren't actually used in the underlying
data are included in the table with a value of \code{0}. If needed, you can
drop unused factor levels by re-running your factor through \code{\link[=factor]{factor()}},
or by calling \code{forcats::fct_drop()}.
}

See the examples section for more information about these properties.
}
\details{
The result of \code{weighted_table()} does not have a \code{"table"} class attached
to it. It is only a double array. This is because "table" objects are
defined as containing integer counts, but weighted tables can utilize
fractional weights.
}
\examples{
x <- factor(c("x", "y", "z", "x", "x", "y"))
y <- factor(c("a", "b", "a", "a", "b", "b"))
w <- c(1.5, 2, 1.1, .5, 3, 2)

weighted_table(x = x, y = y, weights = w)

# ---------------------------------------------------------------------------
# If `weights` contains missing values, then missing values will be
# propagated into the weighted table
x <- factor(c("x", "y", "y"))
y <- factor(c("a", "b", "b"))
w <- c(1, NA, 3)

weighted_table(x = x, y = y, weights = w)

# You can remove the missing values while summing up the weights with
# `na_remove = TRUE`
weighted_table(x = x, y = y, weights = w, na_remove = TRUE)

# ---------------------------------------------------------------------------
# If there are missing values in the factors, those typically don't show
# up in the weighted table
x <- factor(c("x", NA, "y", "x"))
y <- factor(c("a", "b", "a", NA))
w <- 1:4

weighted_table(x = x, y = y, weights = w)

# This is because the missing values aren't considered explicit levels
levels(x)

# You can force them to show up in the table by using `addNA()` ahead of time
# (or `forcats::fct_expand(x, NA)`)
x <- addNA(x, ifany = TRUE)
y <- addNA(y, ifany = TRUE)
levels(x)

weighted_table(x = x, y = y, weights = w)

# ---------------------------------------------------------------------------
# If there are levels in your factors that aren't actually used in the
# underlying data, then they will still show up in the table with a `0` value
x <- factor(c("x", "y", "x"), levels = c("x", "y", "z"))
y <- factor(c("a", "b", "a"), levels = c("a", "b", "c"))
w <- 1:3

weighted_table(x = x, y = y, weights = w)

# If you want to drop these empty factor levels from the result, you can
# rerun `factor()` ahead of time to drop them (or `forcats::fct_drop()`)
x <- factor(x)
y <- factor(y)
levels(x)

weighted_table(x = x, y = y, weights = w)
}
