/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */



/*!
    \page lcmaps_verify_proxy.mod verification plugin

    \section posixsynopsis SYNOPSIS
    \b lcmaps_verify_proxy.mod

    \section posixdesc DESCRIPTION

The Verify Proxy module will verify the validity of the proxy (with its chain)
or any other X.509 certificate chain that has been used in the LCMAPS framework.
The module will report to the LCMAPS framework the validity of the chain.

*/


/*!
    \file   lcmaps_verify_proxy.c
    \brief  Plugin to verify the proxy (and its chain) on its validity.
    \author Oscar Koeroo for the EGEE.
*/


/*****************************************************************************
                            Include header files
******************************************************************************/

/* Needed for (un)setenv and strdup (for latter 500 is sufficient) */
#define	_XOPEN_SOURCE	600

#include "lcmaps_verify_proxy_config.h"

#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <errno.h>
#include <time.h>

#include <openssl/x509.h>
#include <openssl/err.h>

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>


#if defined(HAVE_LCMAPS_LCMAPS_PLUGIN_PROTOTYPES_H)
#   include <lcmaps/lcmaps_plugin_prototypes.h>
#else
#   include "lcmaps_plugin_prototypes.h"
#endif

/* Verification lib */
#include "verify_x509.h"

/* proxylifetime routines */
#include "lcmaps_proxylifetime.h"

/* Define value of environment variable name for allowing discarding of the
 * private key. See man page under --discard_private_key_absence and
 * --never_discard_private_key_absence */
#define DISCARD_PRIVATE_KEY_ABSENCE "VERIFY_PROXY_DISCARD_PRIVATE_KEY_ABSENCE"

#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

/* Define default CA certificate directory */
#ifndef DEFAULT_CERT_DIR
# define DEFAULT_CERT_DIR   "/etc/grid-security/certificates"
#endif

/******************************************************************************
                       Internal prototype
******************************************************************************/
/**
 * actual plugin_run or plugin_verify function, which one depends on last
 * argument
 */
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);


/******************************************************************************
                       Define module specific variables
******************************************************************************/

/* Used for checking the maximum TTLs */
static struct TProxyLevelTTL * plt = NULL;  /* max proxy TTL values	    */
static struct TProxyLevelTTL * vplt = NULL; /* max voms AC TTL values   */

/* commandline options and their defaults */
static int    allow_limited = 1;		/* --allow-limited-proxy */
static int    require_limited = 0;		/* --require-limited-proxy */
static int    only_enforce_lifetime_checks = 0;	/* --only-enforce-lifetime-checks */
static int    discard_private_key_absence = 0;  /* --discard_private_key_absence */
static char * certdir = NULL;			/* -cadir or -certdir */


/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    const char * logstr = "lcmaps_plugin_verify_proxy-plugin_initialize()";
    int i = 0;
    struct stat s;
    int never_discard_private_key_absence = 0;



    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * CERTDIR = The directory which contains the CA certificates
     * VOMSDIR = The directory which contains the certificates of the VOMS servers
     */


    /*
     * Check if a processes has set the option to allow to discard the private
     * key verification. The environment variable that provided this choice will
     * be cleared
     */

    if (getenv (DISCARD_PRIVATE_KEY_ABSENCE))
    {
        /* Env Var found */
        discard_private_key_absence = 1;
    }

    unsetenv (DISCARD_PRIVATE_KEY_ABSENCE);

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        if ( ((strncasecmp(argv[i], "-cadir", (size_t)6) == 0) ||
	      (strncasecmp(argv[i], "-certdir", (size_t)8) == 0) ||
	      (strncasecmp(argv[i], "-capath", (size_t)7) == 0) ||
	      (strncasecmp(argv[i], "--capath", (size_t)8) == 0))
             && (i + 1 < argc)
           )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                if (stat (argv[i + 1], &s) < 0)
                {
                    lcmaps_log(LOG_ERR,"%s: Error: The directory for the CA certificate and CRLs \"%s\" doesn't exist\n", logstr, argv[i + 1]);
                    return LCMAPS_MOD_FAIL;
                }
                certdir = strdup(argv[i + 1]);
            }
            i++;
        }

        else if (strncasecmp(argv[i], "--discard_private_key_absence", (size_t)30) == 0)
        {
            discard_private_key_absence = 1;
        }

        /*
	 * The variable "--never_discard_private_key_absence" will mute the
	 * environment variable that can override the private key verification
	 * functionality.
	 * The environment variable that would allow for the discard of the
	 * check for the private key will be useless.
	 * This is used in situation where the private key check is mandatory
	 * and non-overrideable.
         */
        else if (strncasecmp(argv[i], "--never_discard_private_key_absence", (size_t)36) == 0)
        {
            never_discard_private_key_absence = 1;
        }

        else if (strncasecmp(argv[i], "--allow-limited-proxy", (size_t)21) == 0)
        {
            allow_limited = 1;
        }

        else if (strncasecmp(argv[i], "--disallow-limited-proxy", (size_t)24) == 0)
        {
            allow_limited = 0;
        }

        else if (strncasecmp(argv[i], "--require-limited-proxy", (size_t)23) == 0)
        {
            require_limited = 1;
        }

        else if (strncasecmp(argv[i], "--only-enforce-lifetime-checks", strlen("--only-enforce-lifetime-checks")) == 0)
        {
            only_enforce_lifetime_checks = 1;
        }

        else if (  (strncasecmp(argv[i], "--max-voms-ttl", (size_t)14) == 0) &&
                   (i + 1 < argc) )

        {
            int    level = 0;
            time_t t_for_time = 0;


            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                /* Proxy Time To Life check is activated */

                if ((t_for_time = lcmaps_lifetime_ttl_char2time_t (argv[i + 1])) <= 0)   /* 2d-13:37 */
                {
                    lcmaps_log(LOG_ERR,"%s: Parse error in initialization parameter: %s. Use format: 2d-13:37\n", logstr, argv[i]);
                    return LCMAPS_MOD_FAIL;
                }

                lcmaps_log_debug (5, "%s: VOMS TTL set at: %ld seconds at proxy level: %d\n", logstr, (long)t_for_time, level);
                lcmaps_lifetime_Push_New_Entry (&vplt, level, t_for_time);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: Parse error in initialization parameter: %s. It's NULL string or something undefined.\n", logstr, argv[i + 1]);
                return LCMAPS_MOD_FAIL;
            }
            i++;
        }
        else if ( ((strncasecmp(argv[i], "--max-proxy-level-ttl=", (size_t)22) == 0) && (i + 1 < argc)) ||
                  ((strncasecmp(argv[i], "--max-proxy-level-ttl@", (size_t)22) == 0) && (i + 1 < argc)) )
        {
            int    level = 0;
            time_t t_for_time = 0;
	    long   longval;


            if ( (strlen(argv[i]) == 23) &&
		 ( (argv[i][22] == 'l') || (argv[i][22] == 'L') ) )
                level = LEAF_PROXY;
            else    {
		errno=0;
		longval=strtol(&(argv[i][22]),NULL,10);
		if (errno!=0 || longval<0 || longval>=LEAF_PROXY)  {
		    lcmaps_log(LOG_ERR,"%s: Parse error in initialization parameter: %s.\n", logstr, argv[i]);
		    return LCMAPS_MOD_FAIL;
		}
		level=(int)longval;
	    }

	    if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
	    {
		/* Proxy Time To Life check is activated */
		if ((t_for_time = lcmaps_lifetime_ttl_char2time_t (argv[i + 1])) <= 0)   /* 2d-13:37 */
		{
		    lcmaps_log(LOG_ERR,"%s: Parse error in initialization parameter: %s. Use format: 2d-13:37\n", logstr, argv[i]);
		    return LCMAPS_MOD_FAIL;
		}
		if (level==LEAF_PROXY)
		    lcmaps_log_debug (5,
			"%s: Proxy TTL set at: %ld seconds at LEAF proxy level\n",
			logstr, (long)t_for_time);
		else
		    lcmaps_log_debug (5,
			"%s: Proxy TTL set at: %ld seconds at proxy level: %d\n",
			logstr, (long)t_for_time, level);

		lcmaps_lifetime_Push_New_Entry (&plt, level, t_for_time);
	    }
	    else
	    {
		lcmaps_log(LOG_ERR,"%s: Parse error in initialization parameter: %s. It's NULL string or something undefined.\n", logstr, argv[i + 1]);
		return LCMAPS_MOD_FAIL;
	    }

            i++;
        }

        else
        {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr, argv[i]);
            return LCMAPS_MOD_FAIL;
        }
    }

    /* Set certdir from other means if not specified on the cmdline
     * including gLExec->LCMAPS */
    if (certdir==NULL)	{
	char *value=getenv("X509_CERT_DIR");
	certdir=value ? strdup(value) : strdup(DEFAULT_CERT_DIR);
	if (certdir==NULL)  {
	    lcmaps_log(LOG_ERR,"%s: Out of memory\n",logstr);
	    return LCMAPS_MOD_FAIL;
        }
	lcmaps_log(LOG_INFO,"%s: Using certificate CA Path: %s\n",
		logstr, certdir); 
    }

    /* Finalization check */

    /* Disallowed and required, equals XOR config */
    if ((allow_limited == 0) && (require_limited == 1))
    {
        lcmaps_log(LOG_ERR, "%s: Limited proxy certificates are simultaniously configured to be required and disallowed. Please fix this.\n", logstr);
        return LCMAPS_MOD_FAIL;
    }


    lcmaps_lifetime_Print_TTL_By_Level (plt);
    lcmaps_lifetime_Print_TTL_By_Level (vplt);


    /*
     * The variable "--never_discard_private_key_absence"
     * will mute the environment variable that can override the verification functionality
     * The environment variable that would discard the check for the private key will be useless
     * This is used in situation where the private key check is mandatory.
     */
    if (never_discard_private_key_absence) discard_private_key_absence = 0;

    lcmaps_log_debug(5,"%s: Initialization succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    const char * logstr = "lcmaps_plugin_verify-proxy-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        { "px509_cred"     , "X509 *"                ,  0, NULL},
        { "px509_chain"    , "STACK_OF(X509) *"      ,  0, NULL},
        { "voms_data_list" ,  "lcmaps_vomsdata_t *"  ,  0, NULL},
        { "nvoms_data"     ,  "int"                  ,  0, NULL},
        { "pem_string"     , "char *"                ,  1, NULL},
        { NULL             , NULL                    , -1, NULL}
    };

    lcmaps_log_debug(5,"%s: introspecting\n", logstr);

    *argv = argList;
    lcmaps_log_debug(5,"%s: before lcmaps_cntArgs()\n", logstr);
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: %p\n", logstr, (void*)argList);

    lcmaps_log_debug(3,"%s: Introspect succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}



/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}


/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    const char * logstr = "lcmaps_plugin_verify_proxy-plugin_terminate()";
    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    /* if (vomsdir) free(vomsdir); */
    if (certdir) free(certdir);

    lcmaps_lifetime_FreeProxyLevelTTL (&plt);
    lcmaps_lifetime_FreeProxyLevelTTL (&vplt);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
                       Internal function(s)
******************************************************************************/

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    const char       *  logstr      = "lcmaps_plugin_verify_proxy-plugin_run()";


    STACK_OF(X509)   *  myChain = NULL;
    X509             *  lastCert = NULL;
    proxy_type_t        certificate_type = NONE;
    int                 depth;

    verify_x509_error_t	setparam_result;
    unsigned long       result,errcode;

    char              * pem_string = NULL;
    lcmaps_vomsdata_t * vomsdata  = NULL;
    /* int                 nvomsdata = 0; */

    /* Verification data */
    internal_verify_x509_data_t * verify_data;
    const char       *reason=NULL;


    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_verify_proxy-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_verify_proxy-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR,
            "lcmaps_plugin_voms_verify_proxy-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n",
            lcmaps_mode);
        goto fail_verify_proxy;
    }
    lcmaps_log_debug(5,"%s:\n", logstr);


    /* Fetch the X509 chain from the LCMAPS framework */
    if ( (myChain = *(STACK_OF(X509) **) lcmaps_getArgValue("px509_chain", "STACK_OF(X509) *", argc, argv)) )
    {
        lcmaps_log_debug(5,"%s: found X.509 chain.\n", logstr);
    }
    else
    {
        lcmaps_log(LOG_ERR,"%s: could not get value of X.509 chain.\n", logstr);
        goto fail_verify_proxy;
    }

    /* Fetch the PEM string from the LCMAPS framework */
    if ((pem_string = *(char **) lcmaps_getArgValue("pem_string", "char *", argc, argv)))
    {
        lcmaps_log_debug(5, "%s: found PEM string\n", logstr);
    }
    else
    {
        /* This warning message doesn't make sense if there is no expectency that the private key will need to be used */
        if (!only_enforce_lifetime_checks && !discard_private_key_absence) {
            lcmaps_log(LOG_INFO, "%s: could not get value of PEM string. Used for the private key\n", logstr);
            goto fail_verify_proxy;
        }
        /* This also means that IF a private is supplied that it WILL be
         * checked and that it then MUST match the chain. */
    }


/********************************* New VOMS Structure *********************************/


    vomsdata = NULL;

    if ( ( lcmaps_getArgValue("voms_data_list", "lcmaps_vomsdata_t *", argc, argv)) )
    {
        lcmaps_log_debug(5,"%s: found lcmaps voms_data_list placeholder.\n", logstr);

        if ( !(vomsdata = *(lcmaps_vomsdata_t **) lcmaps_getArgValue("voms_data_list", "lcmaps_vomsdata_t *", argc, argv)) )
        {
            lcmaps_log_debug (3,"%s: value of lcmaps_voms_data_list is empty. No VOMS AC(s) found by the framework in the proxy chain.\n", logstr);
            /* goto fail_verify_proxy; */
        }
        else
        {
            lcmaps_log_debug (5, "%s: vomsdata->nvoms = %d\n", logstr, vomsdata->nvoms);
            lcmaps_log_debug (5, "%s: vomsdata->voms = %p\n", logstr, (void*)vomsdata->voms);
            lcmaps_log_debug (5, "%s: vomsdata->voms[0].userdn = %s\n", logstr, vomsdata->voms[0].user_dn);
        }
    }
    else
    {
        lcmaps_log(LOG_DEBUG,"%s: No value of lcmaps_voms_data_list.\n", logstr);
    }


/********************************* New VOMS Structure *********************************/



    /* If we're not forced to skip the verification of the chain (for speed)
     * Perform Full certificate chain verification
     *
     * Warning! It's less unsecure to run this of no other element is authenticating the credentials!
     */
    if (only_enforce_lifetime_checks == 0)
    {
        /*** verify-lib ***/

        /* The fileName is the proxy file and shown (above tests) to be accessible, including the CA_dir.
         * The CA_dir is used for the CA, sub-CA and CRL files.
         * Note: must implement check that make use of the CA namespace files
         */

	if (verify_X509_init(&verify_data))
            goto fail_verify_proxy;

        setparam_result = verify_X509_setParameter (&verify_data, VERIFY_X509_STACK_OF_X509, myChain);
        if (setparam_result == VER_R_X509_PARAMS_ALREADY_SET)
            lcmaps_log (LOG_WARNING, "%s: VERIFY_X509_CERTIFICATE_FILEPATH already set...\n",logstr);
        else if (setparam_result != VER_R_X509_PARAMS_OK)
        {
            verify_X509_term(&verify_data);
            goto fail_verify_proxy;
        }

        setparam_result = verify_X509_setParameter (&verify_data, VERIFY_X509_CA_PATH, certdir);
        if (setparam_result == VER_R_X509_PARAMS_ALREADY_SET)
            lcmaps_log (LOG_WARNING, "%s: VERIFY_X509_CA_PATH already set...\n",logstr);
        else if (setparam_result != VER_R_X509_PARAMS_OK)
        {
            verify_X509_term(&verify_data);
            goto fail_verify_proxy;
        }


        if (pem_string)
        {
            setparam_result = verify_X509_setParameter (&verify_data, VERIFY_X509_PRIVATEKEY_PEM, pem_string);
            if (setparam_result == VER_R_X509_PARAMS_ALREADY_SET)
                lcmaps_log (LOG_WARNING,"%s: VERIFY_X509_PRIVATEKEY_PEM already set...\n",logstr);
            else if (setparam_result != VER_R_X509_PARAMS_OK)
            {
                verify_X509_term(&verify_data);
                goto fail_verify_proxy;
            }
        }

        /* Allow for the ommission of a private key */
        if (!discard_private_key_absence)
        {
            setparam_result = verify_X509_setParameter (&verify_data, VERIFY_X509_OPTIONS_MUST_HAVE_PRIV_KEY, VERIFY_ENABLE);
            if (setparam_result != VER_R_X509_PARAMS_OK)
            {
                verify_X509_term(&verify_data);
                goto fail_verify_proxy;
            }
        }

        /* Require Limited proxies exclusively to succeed - Redundent / pedantic at this moment */
        if (allow_limited)
            setparam_result = verify_X509_setParameter (&verify_data, VERIFY_X509_OPTIONS_ALLOW_LIMITED_PROXY, VERIFY_ENABLE);
        else
            setparam_result = verify_X509_setParameter (&verify_data, VERIFY_X509_OPTIONS_ALLOW_LIMITED_PROXY, VERIFY_DISABLE);
        if (setparam_result != VER_R_X509_PARAMS_OK)
        {
            verify_X509_term(&verify_data);
            goto fail_verify_proxy;
        }

        /* Require Limited proxies exclusively to succeed - Redundent / pedantic at this moment */
        if (require_limited)
        {
            setparam_result = verify_X509_setParameter (&verify_data, VERIFY_X509_OPTIONS_REQUIRE_LIMITED_PROXY, VERIFY_ENABLE);
            if (setparam_result != VER_R_X509_PARAMS_OK)
            {
                verify_X509_term(&verify_data);
                goto fail_verify_proxy;
            }
        }

	/* returns 0 on success or ERR_get_error() value */
        result = verify_X509_verify(&verify_data);
        verify_X509_term(&verify_data);
        if (result)
            goto fail_verify_proxy;
    }


    /* The final proxy must be a limited proxy */
    /* Require Limited proxies exclusively to succeed */
    if (require_limited)
    {
        /* Select last certificate, check which type of certificate it is, and this MUST match with a limited proxy type */
        lastCert = sk_X509_value(myChain, 0);
        certificate_type = verify_type_of_proxy(lastCert);
        if ( (certificate_type & LIMITED) != LIMITED) {
            lcmaps_log(LOG_NOTICE,
		"%s: The certificate chain does NOT contain a Limited Proxy."
		" This is REQUIRED by the configured policy.\n", logstr);
            goto fail_verify_proxy;
        }
    }


    /*** verify-lib ***/
    depth = sk_X509_num (myChain);

    /* Proxy lifetime checks (yet another post verify check... though!) */
    if (plt)
    {
        lcmaps_log_debug (1,"%s: Enforcing Proxy Life Time Policy\n", logstr);
        if (!lcmaps_lifetime_verifyProxyLifeTime (plt, myChain, depth))
        {
            goto fail_verify_proxy;
        }
    }


    /* VOMS lifetime checks (yet another post verify check... though!) */
    if (vplt)
    {
        lcmaps_log_debug (1,"%s: Enforcing VOMS Life Time Policy.\n", logstr);
        if (!lcmaps_lifetime_verifyVOMSLifeTime (vplt, vomsdata))
        {
            goto fail_verify_proxy;
        }
    }


    /*************************/
    /*************************/

    lcmaps_log(LOG_INFO,"%s: verify proxy plugin succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;

fail_verify_proxy:

    lcmaps_log(LOG_INFO,"%s: verify proxy plugin failed\n", logstr);

    while ((errcode=ERR_get_error())!=0)	{
	reason=ERR_reason_error_string(errcode);
	if (reason)
	    lcmaps_log(LOG_DEBUG,"%s:  error queue: function %s (%s): %s\n",
		    logstr,
		    ERR_func_error_string(errcode),
		    ERR_lib_error_string(errcode),
		    reason);
	else
	    lcmaps_log(LOG_DEBUG,"%s:  error queue: %s\n",
		    logstr,
		    ERR_error_string(errcode,NULL));
    }

    return LCMAPS_MOD_FAIL;
}

