package latexDraw.parsers.svg.elements;

import java.text.ParseException;

import latexDraw.parsers.svg.MalformedSVGDocument;
import latexDraw.parsers.svg.SVGAttributes;
import latexDraw.parsers.svg.SVGDocument;
import latexDraw.parsers.svg.SVGElements;
import latexDraw.parsers.svg.parsers.SVGLengthParser;

import org.w3c.dom.Node;

/**
 * Defines the SVG tag <code>ciricle</code>.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 10/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 * @since 0.1
 */
public class SVGCircleElement extends SVGElement
{

	public SVGCircleElement(Node n, SVGElement p) throws MalformedSVGDocument
	{
		super(n, p);
	}
	
	
	/**
	 * Creates an SVG circle element.
	 * @param cx The X-centre coordinate.
	 * @param cy The Y-centre coordinate.
	 * @param r The radius of the circle.
	 * @param owner The document owner.
	 * @throws IllegalArgumentException If owner is null or if a given value is not valid.
	 * @since 0.1
	 */
	public SVGCircleElement(double cx, double cy, double r, SVGDocument owner)
	{
		super(owner);
		
		setAttribute(SVGAttributes.SVG_CX, String.valueOf(cx));
		setAttribute(SVGAttributes.SVG_CY, String.valueOf(cy));
		setAttribute(SVGAttributes.SVG_R, String.valueOf(r));
		setNodeName(SVGElements.SVG_CIRCLE);
		ownerDocument = owner;
		
		if(!checkAttributes())
			throw new IllegalArgumentException();
	}
	
	
	
	/**
	 * Creates a circle with a radius equal to 0.
	 * @param doc The owner document.
	 * @since 0.1
	 */
	public SVGCircleElement(SVGDocument doc)
	{
		super(doc);
		
		setNodeName(SVGElements.SVG_CIRCLE);
		setAttribute(SVGAttributes.SVG_R, "0");//$NON-NLS-1$
	}



	/**
	 * @return The x-axis coordinate of the centre of the circle (0 if there it does not exist or it is not a coordinate).
	 * @since 0.1
	 */
	public double getCx()
	{
		String v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_CX);
		double cx;
		
		try { cx = v==null ? 0 : new SVGLengthParser(v).parseCoordinate().getValue(); }
		catch(ParseException e) { cx = 0; }
		
		return cx;
	}
	
	
	
	/**
	 * @return The y-axis coordinate of the centre of the circle (0 if there it does not exist or it is not a coordinate).
	 * @since 0.1
	 */
	public double getCy()
	{
		String v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_CY);
		double cy;
		
		try { cy = v==null ? 0 : new SVGLengthParser(v).parseCoordinate().getValue(); }
		catch(ParseException e) { cy = 0; }
		
		return cy;
	}
	
	
	
	/**
	 * @return The radius of the circle (NaN if there it does not exist or it is not a length).
	 * @since 0.1
	 */
	public double getR()
	{
		String v = getAttribute(getUsablePrefix()+SVGAttributes.SVG_R);
		double r;
		
		try { r = v==null ? Double.NaN : new SVGLengthParser(v).parseLength().getValue(); }
		catch(ParseException e) { r = Double.NaN; }
		
		return r;
	}
	
	
	
	/**
	 * Sets the X-coordinate of the circle.
	 * @param cx The new X-coordinate of the circle.
	 * @since 0.1
	 */
	public void setCx(double cx)
	{
		setAttribute(getUsablePrefix()+SVGAttributes.SVG_CX, String.valueOf(cx));
	}
	
	
	
	
	/**
	 * Sets the Y-coordinate of the circle.
	 * @param cy The new Y-coordinate of the circle.
	 * @since 0.1
	 */
	public void setCy(Double cy)
	{
		setAttribute(getUsablePrefix()+SVGAttributes.SVG_CY, String.valueOf(cy));
	}
	
	
	
	/**
	 * Sets the radius of the circle.
	 * @param width The new radius of the circle.
	 * @since 0.1
	 */
	public void setR(double width)
	{
		if(width>=0)
			setAttribute(getUsablePrefix()+SVGAttributes.SVG_R, String.valueOf(width));
	}
	
	
	@Override
	public boolean checkAttributes()
	{
		double r = getR();
		
		return !Double.isNaN(r) && r>=0;
	}



	@Override
	public boolean enableRendering()
	{
		return getR()!=0 ;
	}
}
