/** @file
  Application for X509 Cert Validation.

Copyright (c) 2022, Intel Corporation. All rights reserved.<BR>
SPDX-License-Identifier: BSD-2-Clause-Patent

**/

#include "TestBaseCryptLib.h"

GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8  mOidSubjectAltName[] = { 0x55, 0x1D, 0x11 };

//
// use openssl tool to create the test certificates.
//
// openssl req -nodes -x509 -days 3650 -newkey rsa:4096 -keyout ca.key -out ca.cert -sha256 -subj "/CN=intel test RSA CA"
// openssl rsa -in ca.key -outform der -out ca.key.der
// openssl req -nodes -newkey rsa:3072 -keyout inter.key -out inter.req -sha256 -batch -subj "/CN=intel test RSA intermediate cert"
// openssl req -nodes -newkey rsa:2048 -keyout end_requester.key -out end_requester.req -sha256 -batch -subj "/CN=intel test RSA requseter cert"
// openssl req -nodes -newkey rsa:2048 -keyout end_responder.key -out end_responder.req -sha256 -batch -subj "/CN=intel test RSA responder cert"
// openssl x509 -req -in inter.req -out inter.cert -CA ca.cert -CAkey ca.key -sha256 -days 3650 -set_serial 1 -extensions v3_inter -extfile ../openssl.cnf
// openssl x509 -req -in end_requester.req -out end_requester.cert -CA inter.cert -CAkey inter.key -sha256 -days 3650 -set_serial 2 -extensions v3_end -extfile ../openssl.cnf
// openssl x509 -req -in end_responder.req -out end_responder.cert -CA inter.cert -CAkey inter.key -sha256 -days 3650 -set_serial 3 -extensions v3_end -extfile ../openssl.cnf
// openssl asn1parse -in ca.cert -out ca.cert.der
// openssl asn1parse -in inter.cert -out inter.cert.der
// openssl asn1parse -in end_requester.cert -out end_requester.cert.der
// cat ca.cert.der inter.cert.der end_requester.cert.der > bundle_requester.certchain.der
// openssl rsa -inform PEM -outform DER -in end_requester.key -out end_requester.key.der
//

//
// dump of inter.cert.der
//
GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8  mTestCert[] = {
  0x30, 0x82, 0x04, 0xa0, 0x30, 0x82, 0x02, 0x88, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x01,
  0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30,
  0x1c, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x11, 0x69, 0x6e, 0x74, 0x65,
  0x6c, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x43, 0x41, 0x30, 0x1e, 0x17,
  0x0d, 0x32, 0x32, 0x30, 0x31, 0x30, 0x35, 0x30, 0x36, 0x31, 0x31, 0x30, 0x36, 0x5a, 0x17, 0x0d,
  0x33, 0x32, 0x30, 0x31, 0x30, 0x33, 0x30, 0x36, 0x31, 0x31, 0x30, 0x36, 0x5a, 0x30, 0x2b, 0x31,
  0x29, 0x30, 0x27, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6c, 0x20,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65,
  0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x63, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d,
  0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01,
  0x8f, 0x00, 0x30, 0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xd1, 0xcf, 0x2d, 0x21, 0x38,
  0xe4, 0x81, 0xa6, 0xf2, 0xe5, 0xca, 0xac, 0x09, 0x2b, 0x04, 0x2a, 0x52, 0x76, 0x2f, 0xa3, 0x4b,
  0x43, 0xc2, 0xd6, 0xb5, 0x3e, 0x2b, 0x29, 0x6f, 0x1b, 0x7d, 0x36, 0x60, 0xcd, 0x71, 0xf4, 0xcf,
  0x8a, 0x24, 0xac, 0x8f, 0x5f, 0x78, 0x70, 0xc9, 0x84, 0x48, 0xfb, 0x1d, 0xcb, 0xee, 0x34, 0x3c,
  0x4c, 0xe1, 0x1a, 0x92, 0x54, 0x43, 0x45, 0xe0, 0xdd, 0x14, 0xd5, 0xac, 0x13, 0xae, 0xcc, 0x45,
  0xdc, 0x52, 0xdd, 0x45, 0x22, 0x10, 0xab, 0xb2, 0x1b, 0x3e, 0xad, 0x36, 0x6f, 0x00, 0x59, 0xb6,
  0xfd, 0x27, 0xb8, 0x57, 0xbb, 0x66, 0x72, 0x1e, 0x7b, 0xe8, 0xc7, 0xaf, 0x5e, 0x65, 0x5c, 0x58,
  0x3d, 0x9e, 0x78, 0x26, 0xb7, 0x70, 0x9a, 0xe7, 0x3c, 0xfc, 0x90, 0x64, 0xb2, 0xb2, 0xb2, 0x80,
  0x22, 0xb3, 0x1c, 0x23, 0x73, 0x6a, 0xe3, 0xcb, 0x5b, 0x4e, 0x29, 0x71, 0xdd, 0xfd, 0xd7, 0x15,
  0x3a, 0xd6, 0xa3, 0xe0, 0x4a, 0x53, 0x65, 0xa6, 0xae, 0x6a, 0x83, 0xb0, 0x00, 0x3f, 0x64, 0x4b,
  0x01, 0xcd, 0x97, 0xb1, 0xe1, 0x54, 0x1e, 0xa2, 0x33, 0x4e, 0xf0, 0x13, 0x33, 0x80, 0xe5, 0x41,
  0x9c, 0x5e, 0xb3, 0x5d, 0xae, 0x53, 0x2c, 0x41, 0x51, 0xf2, 0x00, 0x3e, 0x03, 0xe2, 0x0d, 0xa4,
  0xf5, 0x24, 0x08, 0x46, 0xc2, 0xed, 0x6b, 0xf2, 0xe0, 0x1b, 0x04, 0xb7, 0xe8, 0xda, 0x85, 0xa4,
  0x2f, 0x4b, 0x53, 0xfa, 0x76, 0x72, 0x0a, 0x28, 0xbf, 0x3c, 0xa8, 0x85, 0x03, 0xee, 0x51, 0xb5,
  0x5c, 0xa9, 0xa5, 0x70, 0x47, 0x07, 0x11, 0x32, 0xc5, 0x5f, 0x7a, 0x07, 0xfb, 0x5f, 0x4a, 0x52,
  0x65, 0x33, 0x0c, 0x17, 0x8e, 0x74, 0xe5, 0xec, 0x7d, 0x52, 0xa0, 0xd4, 0xef, 0x64, 0xcc, 0x5e,
  0xe2, 0x3d, 0x46, 0x1e, 0x73, 0xab, 0x43, 0x2d, 0x05, 0xfa, 0xa3, 0x6d, 0x86, 0x02, 0xe7, 0xc8,
  0x5f, 0x44, 0xfe, 0xeb, 0x14, 0x73, 0x2d, 0xf4, 0x0b, 0x72, 0x49, 0xe3, 0xef, 0x55, 0x05, 0x1b,
  0x2f, 0xf9, 0x73, 0xeb, 0xed, 0xcc, 0x5e, 0x52, 0x17, 0x43, 0x94, 0x85, 0xc7, 0x0e, 0x73, 0x89,
  0x22, 0xdc, 0x02, 0x5f, 0x99, 0x69, 0x58, 0x16, 0x89, 0x1e, 0x6f, 0x75, 0xf8, 0xd3, 0xe7, 0x0a,
  0xb1, 0x01, 0x42, 0x71, 0xa6, 0xc5, 0xb2, 0xd0, 0xc0, 0x75, 0x96, 0x2e, 0xc2, 0x32, 0x49, 0x0a,
  0xe8, 0x85, 0xbd, 0x22, 0x4c, 0x95, 0xd4, 0xf8, 0x2d, 0x72, 0x05, 0x4f, 0x0e, 0x56, 0x9c, 0xc3,
  0x30, 0x3d, 0xfd, 0x30, 0x63, 0x92, 0x72, 0x6b, 0xaf, 0x80, 0x9d, 0xd0, 0xc2, 0x36, 0xe7, 0xc1,
  0x37, 0xd0, 0x64, 0x88, 0x64, 0x3c, 0x33, 0xdd, 0x87, 0xdd, 0xd9, 0xa2, 0x79, 0x44, 0xd8, 0x25,
  0x84, 0x0e, 0xfd, 0x9d, 0xd6, 0xc1, 0x7a, 0x2d, 0x23, 0x08, 0x9d, 0x02, 0x03, 0x01, 0x00, 0x01,
  0xa3, 0x5e, 0x30, 0x5c, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01,
  0x01, 0xff, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x01, 0xfe, 0x30,
  0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x46, 0xcd, 0xb4, 0x52, 0x02, 0x29,
  0x26, 0xc7, 0x52, 0x23, 0x81, 0x65, 0xcd, 0x87, 0x2f, 0x96, 0x1b, 0x01, 0x2c, 0xc5, 0x30, 0x20,
  0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06,
  0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02,
  0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03,
  0x82, 0x02, 0x01, 0x00, 0x71, 0x14, 0xca, 0xfe, 0x03, 0x92, 0xc6, 0x88, 0x52, 0x76, 0x43, 0xd9,
  0x8a, 0xf7, 0x22, 0x94, 0xef, 0x40, 0xa7, 0xc3, 0x7b, 0x6f, 0xb4, 0xd0, 0x42, 0xc3, 0x59, 0x1f,
  0x77, 0xef, 0xac, 0x18, 0xce, 0x7c, 0xab, 0x48, 0x85, 0x22, 0xf1, 0x50, 0x2c, 0xe0, 0x63, 0x4d,
  0x9d, 0x0e, 0xed, 0x38, 0xff, 0xd9, 0xed, 0x3c, 0x7e, 0x3e, 0x84, 0xb1, 0xa7, 0x78, 0x62, 0x37,
  0x39, 0x6d, 0xd5, 0x5d, 0x46, 0x9a, 0x6f, 0x63, 0xc2, 0xa7, 0x90, 0xf9, 0x7a, 0xa7, 0x4b, 0xa2,
  0x8a, 0xa8, 0x50, 0x13, 0x16, 0x9c, 0x4f, 0xcc, 0x28, 0x58, 0x6c, 0x9f, 0x65, 0x6e, 0xf0, 0x9a,
  0x92, 0x59, 0x94, 0xa4, 0xb5, 0xfd, 0x8f, 0xf0, 0x7c, 0x1e, 0xba, 0xc3, 0x21, 0x13, 0xfb, 0x33,
  0xba, 0xad, 0x4b, 0x1d, 0x42, 0x23, 0x16, 0x8b, 0x3e, 0xcd, 0xb5, 0x69, 0x9c, 0x7c, 0xa1, 0x9b,
  0x92, 0xc1, 0x2f, 0x9c, 0x89, 0x3f, 0x28, 0x24, 0x67, 0x48, 0x47, 0xf0, 0xb1, 0xc0, 0xf0, 0x4c,
  0xbf, 0xf6, 0x64, 0xde, 0x33, 0xc3, 0xd6, 0x5b, 0xe6, 0xc3, 0xdd, 0xa5, 0xba, 0x42, 0x02, 0x0f,
  0xbc, 0xbe, 0xde, 0x14, 0x12, 0x7c, 0xfb, 0xe6, 0xbb, 0xdd, 0xf2, 0x6b, 0xd7, 0x75, 0xe6, 0xf5,
  0xda, 0xbe, 0xe9, 0xbf, 0xa6, 0x87, 0x09, 0x41, 0x6c, 0x9d, 0x19, 0x9c, 0xb1, 0x7c, 0x89, 0x9f,
  0x32, 0xb3, 0x0a, 0x59, 0x10, 0xc6, 0x8c, 0xa0, 0x5b, 0xda, 0xa6, 0xa3, 0xf8, 0x33, 0x77, 0x7a,
  0xf4, 0xb0, 0x8a, 0xcf, 0xc8, 0x3f, 0x2e, 0xb8, 0xf1, 0x9e, 0x7b, 0x47, 0x2d, 0x85, 0x3b, 0x58,
  0xcd, 0x4b, 0xbc, 0x87, 0xd2, 0xf3, 0xcd, 0x36, 0x80, 0x9b, 0x4a, 0x0f, 0x24, 0x30, 0x1d, 0x3f,
  0x6f, 0x56, 0xcf, 0x6e, 0x2d, 0xcf, 0xa8, 0x17, 0xd5, 0x97, 0x0b, 0x22, 0xa6, 0x59, 0xef, 0xef,
  0xd4, 0x9a, 0x0e, 0x6f, 0xb9, 0xf0, 0x48, 0x2a, 0x54, 0x22, 0x77, 0x27, 0x90, 0x84, 0x42, 0x56,
  0x85, 0x80, 0x78, 0x4f, 0xd9, 0x14, 0x2a, 0xf0, 0x5d, 0x6b, 0x46, 0x60, 0x3d, 0xad, 0xa5, 0xa2,
  0xb9, 0x04, 0x23, 0x92, 0x1b, 0x70, 0xa3, 0xdb, 0xaa, 0xf0, 0x5f, 0x1c, 0xd8, 0x26, 0xbb, 0x51,
  0x17, 0xfc, 0x93, 0x6d, 0x70, 0x19, 0x54, 0xe2, 0x6f, 0x82, 0x8a, 0x49, 0xa6, 0x19, 0xcc, 0x97,
  0x53, 0x90, 0x27, 0xd9, 0x8d, 0xaa, 0x8c, 0xc3, 0x2b, 0xbc, 0x0a, 0x72, 0x3a, 0x41, 0xb8, 0xfa,
  0x1e, 0xbb, 0x8b, 0x27, 0x06, 0x75, 0x53, 0x91, 0xac, 0x8d, 0x75, 0xf2, 0xa6, 0xea, 0x85, 0x7e,
  0x34, 0x06, 0x9e, 0xf9, 0xe9, 0x13, 0xa3, 0xfe, 0x2e, 0x38, 0x4b, 0x3f, 0x61, 0x11, 0x1f, 0x6b,
  0xd3, 0xfe, 0xc9, 0x13, 0xbc, 0x12, 0xfe, 0xbc, 0xff, 0xaa, 0x38, 0x73, 0x8a, 0x73, 0x8c, 0xcf,
  0xcc, 0xd3, 0xe7, 0xba, 0x98, 0xfd, 0xf4, 0xf5, 0xf6, 0xa9, 0xc8, 0x07, 0x5b, 0x16, 0xae, 0x76,
  0x86, 0x24, 0x25, 0xcc, 0x7c, 0xdb, 0x1e, 0x7f, 0xc4, 0xe4, 0xd3, 0x89, 0x5a, 0x91, 0x6a, 0x9c,
  0x93, 0x03, 0xd2, 0xd2, 0xc0, 0x29, 0x69, 0x58, 0x52, 0x6c, 0xca, 0x91, 0x7d, 0xd2, 0x3d, 0xc7,
  0x2f, 0x8a, 0x4e, 0x55, 0x03, 0x34, 0x16, 0x01, 0x2f, 0x4f, 0x57, 0xea, 0x10, 0xa7, 0xfb, 0xe3,
  0x2e, 0x2c, 0x7f, 0x5d, 0x27, 0x93, 0x74, 0x95, 0x25, 0x1a, 0x1e, 0x54, 0x1a, 0xb6, 0xe1, 0xdc,
  0xc8, 0xe4, 0x84, 0xeb, 0x38, 0x0f, 0x05, 0x74, 0xf3, 0x3a, 0xf9, 0xd8, 0x00, 0x43, 0xe9, 0x24,
  0x1e, 0x45, 0xb4, 0xe2, 0x38, 0x8b, 0x6a, 0x4f, 0xb6, 0x0b, 0x1d, 0xac, 0xbc, 0xec, 0xda, 0x41,
  0x9c, 0x7f, 0xa8, 0x15, 0x09, 0x3b, 0x0b, 0x99, 0xc4, 0x48, 0xdd, 0xde, 0x0d, 0x65, 0x86, 0x72,
  0xaa, 0x0a, 0x4e, 0x71,
};

//
// dump of ca.cert.der
//
GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8  mTestCaCert[] = {
  0x30, 0x82, 0x05, 0x19, 0x30, 0x82, 0x03, 0x01, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x14, 0x1d,
  0xaf, 0x4e, 0xf1, 0x99, 0xbf, 0xee, 0xe2, 0x6d, 0x7e, 0x4a, 0xac, 0xac, 0x20, 0x00, 0xeb, 0x78,
  0xf6, 0x6a, 0xe9, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
  0x05, 0x00, 0x30, 0x1c, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x11, 0x69,
  0x6e, 0x74, 0x65, 0x6c, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x43, 0x41,
  0x30, 0x1e, 0x17, 0x0d, 0x32, 0x32, 0x30, 0x31, 0x30, 0x35, 0x30, 0x36, 0x31, 0x31, 0x30, 0x35,
  0x5a, 0x17, 0x0d, 0x33, 0x32, 0x30, 0x31, 0x30, 0x33, 0x30, 0x36, 0x31, 0x31, 0x30, 0x35, 0x5a,
  0x30, 0x1c, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x11, 0x69, 0x6e, 0x74,
  0x65, 0x6c, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x43, 0x41, 0x30, 0x82,
  0x02, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
  0x00, 0x03, 0x82, 0x02, 0x0f, 0x00, 0x30, 0x82, 0x02, 0x0a, 0x02, 0x82, 0x02, 0x01, 0x00, 0xd5,
  0x13, 0x55, 0xb1, 0x0d, 0x44, 0x17, 0x6a, 0x06, 0x39, 0x66, 0xee, 0xbc, 0x43, 0x81, 0xea, 0xe0,
  0x93, 0x3f, 0x70, 0xb5, 0x5b, 0x07, 0xde, 0xa0, 0x24, 0x47, 0xd0, 0xf8, 0x99, 0xe6, 0x50, 0x17,
  0xb4, 0x02, 0x60, 0xd9, 0x1a, 0x6c, 0xd3, 0x55, 0x60, 0x61, 0xde, 0x80, 0x6d, 0x38, 0x29, 0x46,
  0xbb, 0x45, 0xa2, 0xeb, 0x24, 0x1c, 0xbb, 0x87, 0x1f, 0x14, 0xed, 0x3a, 0xd5, 0x32, 0xd5, 0x11,
  0x4b, 0xf4, 0x99, 0x78, 0x00, 0x11, 0x8d, 0x6d, 0x6e, 0x5a, 0x0b, 0x81, 0xff, 0x08, 0x59, 0x01,
  0x2b, 0xf0, 0x9b, 0x52, 0x88, 0x41, 0xb6, 0x3e, 0x62, 0x7b, 0xbd, 0x25, 0x96, 0x64, 0x8e, 0xb1,
  0x2b, 0x6a, 0xf8, 0x39, 0x93, 0x0a, 0x6b, 0x3f, 0x8c, 0x2b, 0x35, 0x6e, 0x80, 0x32, 0x86, 0xf6,
  0x0f, 0x1e, 0x8c, 0xd4, 0x4e, 0x77, 0xbe, 0xaf, 0xe5, 0x4c, 0x47, 0x66, 0xdd, 0x62, 0xce, 0x64,
  0x65, 0x80, 0xca, 0xbd, 0x87, 0x94, 0x3e, 0xff, 0x3d, 0x0a, 0xd4, 0x39, 0xcc, 0xd1, 0xef, 0xd1,
  0x1a, 0x55, 0x2c, 0xeb, 0xdc, 0xcf, 0x9a, 0x59, 0x92, 0xda, 0xca, 0xf6, 0xae, 0x88, 0xbc, 0xc0,
  0x43, 0xe5, 0xa9, 0x60, 0xb2, 0x0f, 0x69, 0xa2, 0x2a, 0x98, 0x56, 0x0f, 0x4b, 0x0b, 0x63, 0xb3,
  0x4e, 0x8b, 0x43, 0xb4, 0x7f, 0x52, 0x1e, 0x5a, 0x9e, 0xdf, 0xa7, 0x98, 0x4d, 0xb9, 0x3b, 0x58,
  0x80, 0x69, 0x38, 0xb7, 0xeb, 0x20, 0x91, 0xbf, 0x59, 0x81, 0x44, 0xfc, 0xc6, 0xaf, 0xc2, 0xc8,
  0xe1, 0xae, 0x3d, 0xc5, 0xb6, 0x55, 0xa8, 0xa2, 0xcd, 0xec, 0x0b, 0xbe, 0x9c, 0x34, 0x2b, 0xf5,
  0x4b, 0x2e, 0x3b, 0x26, 0xaa, 0x22, 0x12, 0x29, 0xc2, 0x66, 0xc2, 0x40, 0xdd, 0x12, 0xfa, 0xf0,
  0xb4, 0xe8, 0x95, 0x20, 0x59, 0x91, 0x9e, 0x7d, 0x36, 0x6e, 0x2d, 0x99, 0x53, 0x44, 0x2b, 0x43,
  0x12, 0x08, 0x74, 0x76, 0xd5, 0x21, 0xd3, 0x08, 0x37, 0xe1, 0xe6, 0xe3, 0xc4, 0x67, 0x88, 0x29,
  0xea, 0x06, 0x97, 0x9f, 0x61, 0x0d, 0x5d, 0x39, 0x17, 0x03, 0x87, 0x89, 0x35, 0xe0, 0xd5, 0x87,
  0xa1, 0x5b, 0x0c, 0x3c, 0x35, 0x3d, 0xd6, 0x6f, 0xdd, 0x6f, 0xa5, 0x11, 0x9f, 0x80, 0x20, 0xca,
  0x8b, 0x67, 0xd2, 0x2c, 0x7c, 0x51, 0x68, 0x15, 0x2c, 0x9c, 0x08, 0x86, 0xc8, 0x6c, 0xf0, 0x9d,
  0xc2, 0x98, 0xb9, 0x6e, 0x31, 0xaf, 0x2b, 0x34, 0x9c, 0xe3, 0xd0, 0xe9, 0x85, 0x37, 0x2b, 0x1f,
  0x21, 0xd2, 0xb9, 0xa4, 0x8d, 0x3f, 0xa3, 0x1a, 0x9f, 0x7d, 0x1b, 0xb6, 0xe0, 0xf6, 0x19, 0xf4,
  0x19, 0xf8, 0x50, 0xa7, 0x1b, 0xec, 0xb9, 0xbb, 0xa4, 0xd2, 0x20, 0xf3, 0xed, 0xc7, 0xa7, 0x94,
  0x2b, 0x0b, 0x9a, 0x74, 0xb8, 0xba, 0xd7, 0x42, 0x97, 0xab, 0xb4, 0xe9, 0xbe, 0x1e, 0x07, 0x0c,
  0x05, 0x55, 0x54, 0xba, 0x05, 0x73, 0x99, 0xf2, 0x29, 0x88, 0x69, 0x4f, 0x1b, 0x2e, 0x87, 0x7b,
  0x74, 0x94, 0xae, 0xc2, 0x2f, 0x3c, 0x90, 0xb5, 0x5c, 0x01, 0x46, 0xc6, 0x42, 0xfb, 0xa9, 0xad,
  0x82, 0x1b, 0xd9, 0x38, 0xbe, 0x0d, 0x72, 0x4c, 0x06, 0x03, 0x92, 0xc2, 0xc8, 0x7f, 0x7b, 0xd4,
  0x86, 0x83, 0xb8, 0x62, 0xd9, 0xf7, 0x2f, 0xc1, 0x57, 0x88, 0x85, 0xa5, 0x61, 0xc5, 0x93, 0xfc,
  0x17, 0xe0, 0x98, 0x11, 0x88, 0x0d, 0x03, 0x0b, 0x6e, 0x01, 0x16, 0xcc, 0xf2, 0x3c, 0x5f, 0x61,
  0x8c, 0x27, 0x31, 0xca, 0x4d, 0x35, 0x7a, 0xc1, 0xf3, 0xfe, 0x06, 0xff, 0xca, 0x7b, 0x96, 0xad,
  0x0d, 0xff, 0x9e, 0x12, 0x44, 0x51, 0x6e, 0xf4, 0xaa, 0x2e, 0xce, 0xe8, 0xc7, 0xa6, 0xe9, 0xae,
  0xc0, 0xbe, 0xde, 0x7b, 0x93, 0x3d, 0x1d, 0xf9, 0xfb, 0x4d, 0xa7, 0x0b, 0x73, 0xfb, 0x7b, 0x02,
  0x03, 0x01, 0x00, 0x01, 0xa3, 0x53, 0x30, 0x51, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04,
  0x16, 0x04, 0x14, 0x03, 0xfe, 0xb0, 0x3f, 0x09, 0xe5, 0xe1, 0xf4, 0x2e, 0x87, 0xfb, 0x41, 0x86,
  0xf6, 0x76, 0xd3, 0x93, 0xbe, 0x96, 0x85, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
  0x30, 0x16, 0x80, 0x14, 0x03, 0xfe, 0xb0, 0x3f, 0x09, 0xe5, 0xe1, 0xf4, 0x2e, 0x87, 0xfb, 0x41,
  0x86, 0xf6, 0x76, 0xd3, 0x93, 0xbe, 0x96, 0x85, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
  0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
  0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01, 0x00, 0x02, 0x15, 0x73,
  0x60, 0x10, 0xbc, 0x95, 0xac, 0x63, 0x15, 0x09, 0x6b, 0x0c, 0x09, 0x4a, 0x74, 0xcd, 0xe3, 0x58,
  0xcc, 0x6c, 0xf3, 0x55, 0x80, 0xe9, 0x75, 0x47, 0x2c, 0x80, 0x4e, 0x5f, 0xfe, 0xbd, 0x2e, 0xd7,
  0x42, 0x89, 0x44, 0x57, 0xbe, 0x51, 0x84, 0x53, 0xb6, 0x40, 0x80, 0xd6, 0xcc, 0xe2, 0x80, 0x22,
  0x53, 0xd9, 0x31, 0x65, 0xf9, 0x3c, 0x8b, 0x7b, 0xe6, 0xa6, 0x6e, 0xfd, 0x9d, 0x27, 0xe5, 0xcd,
  0xfd, 0x82, 0xcf, 0xd6, 0x18, 0xbe, 0xa6, 0xed, 0x72, 0xce, 0x5f, 0x45, 0x12, 0x6a, 0xaa, 0x95,
  0x42, 0x25, 0x28, 0x8b, 0xfc, 0x4b, 0xc9, 0xad, 0xdd, 0xdb, 0x4c, 0x74, 0x10, 0x2e, 0x90, 0x2a,
  0x06, 0x4f, 0x2e, 0xb4, 0x54, 0xb9, 0xc0, 0x60, 0xb9, 0x4d, 0xee, 0x59, 0x1c, 0x18, 0x8d, 0xd1,
  0x49, 0xc3, 0xe9, 0x1b, 0xf1, 0xfc, 0xc3, 0x83, 0x1f, 0x6a, 0xb0, 0xfc, 0x8d, 0xfc, 0x30, 0xed,
  0x9c, 0xcb, 0x78, 0x52, 0xe8, 0x09, 0x3d, 0x4f, 0xdc, 0xbc, 0xad, 0x84, 0xc1, 0xd5, 0x5a, 0x0a,
  0x07, 0xa3, 0xf6, 0x42, 0xd7, 0x54, 0x55, 0x01, 0x8e, 0x53, 0xce, 0xcb, 0x2a, 0x11, 0xf7, 0x89,
  0x7e, 0xaf, 0x6f, 0x4c, 0xb9, 0x56, 0x4a, 0x67, 0x4a, 0xf9, 0x4f, 0x64, 0x15, 0xfa, 0xb0, 0xf9,
  0x97, 0xe2, 0xf6, 0xa8, 0xf4, 0xe7, 0x0a, 0x7a, 0x83, 0x4e, 0xf6, 0xe9, 0xac, 0x5e, 0xd9, 0xa8,
  0xea, 0x6b, 0x06, 0xcb, 0x2c, 0x41, 0xc1, 0x7e, 0xf5, 0x79, 0xfc, 0x7c, 0x05, 0x06, 0x8f, 0x27,
  0xaa, 0x3b, 0x61, 0x82, 0x72, 0x55, 0xa9, 0xa0, 0xa0, 0xa5, 0x69, 0x2f, 0x95, 0x40, 0xfc, 0xfe,
  0x4a, 0x0f, 0x7d, 0x8c, 0x89, 0xaa, 0xc0, 0x1d, 0x87, 0x03, 0xa1, 0xce, 0xee, 0x23, 0x4f, 0xc5,
  0x7c, 0xb4, 0xb6, 0x2b, 0x6f, 0x05, 0x30, 0xc9, 0x16, 0x51, 0xdf, 0xc7, 0x16, 0x3c, 0x08, 0x38,
  0x20, 0xf9, 0xc5, 0xe0, 0x4a, 0xfa, 0xcb, 0x8c, 0xc3, 0xc5, 0xbb, 0x5c, 0xad, 0xca, 0xc2, 0x52,
  0x45, 0x2f, 0x54, 0x70, 0x78, 0x33, 0x70, 0xc2, 0xed, 0x68, 0xf1, 0x89, 0x67, 0xa3, 0x19, 0x24,
  0xcb, 0x8f, 0x99, 0x1b, 0x28, 0x81, 0x6c, 0x4e, 0x25, 0xb1, 0x27, 0x3d, 0x9f, 0xe7, 0x3d, 0xa7,
  0x73, 0x9e, 0x4c, 0x1a, 0x63, 0x8e, 0xf9, 0xa7, 0xb6, 0x21, 0xe7, 0x4c, 0xdf, 0xfb, 0x36, 0x47,
  0xda, 0x2d, 0xbb, 0x52, 0x55, 0xf8, 0x44, 0x0d, 0x0c, 0xde, 0xe2, 0x13, 0x42, 0x1b, 0xa2, 0xad,
  0xa0, 0x0f, 0x39, 0x6c, 0x78, 0x32, 0x7a, 0x03, 0x9e, 0x55, 0x4e, 0x43, 0xf7, 0x0c, 0x35, 0xd9,
  0x1d, 0x2c, 0x0f, 0x30, 0x30, 0x3e, 0x09, 0xe2, 0x31, 0xa6, 0xb0, 0x1e, 0xa9, 0xf5, 0x4b, 0xa1,
  0x74, 0x09, 0x50, 0xd4, 0xd3, 0xd3, 0x3e, 0x76, 0xb1, 0x67, 0xfc, 0x51, 0xb0, 0x93, 0x22, 0xc4,
  0x6b, 0x8a, 0x27, 0x45, 0x19, 0x3b, 0x35, 0x91, 0x61, 0x36, 0xe7, 0x9c, 0xf6, 0xda, 0x96, 0x30,
  0x7d, 0xf4, 0x11, 0xc4, 0x3f, 0x35, 0x4e, 0x7a, 0xd6, 0x6e, 0xb6, 0xea, 0xe7, 0x66, 0x6f, 0x23,
  0x5c, 0x53, 0x76, 0x53, 0xfc, 0x35, 0x93, 0xe5, 0xfc, 0xb9, 0x6b, 0xb9, 0xd3, 0x6c, 0x48, 0x66,
  0x6f, 0xd7, 0x10, 0x6e, 0x25, 0x72, 0x71, 0x31, 0xfa, 0xc0, 0xdf, 0x31, 0xca, 0xd1, 0xaf, 0xc2,
  0x8e, 0xa5, 0xca, 0xd7, 0x3f, 0x4e, 0xde, 0x47, 0xd5, 0x8e, 0xfc, 0x75, 0xb6, 0x71, 0x83, 0xd9,
  0xfd, 0x11, 0x35, 0x81, 0xbf, 0x10, 0x0d, 0x3e, 0x50, 0x45, 0x07, 0x39, 0x08, 0x73, 0x7a, 0x0b,
  0x21, 0x32, 0xaf, 0xf4, 0x99, 0xeb, 0x4d, 0xd4, 0xe8, 0x2a, 0x06, 0x98, 0x43, 0xbb, 0xbb, 0x11,
  0x63, 0x99, 0xa8, 0x41, 0x22, 0xe8, 0x86, 0x79, 0x4b, 0x53, 0xb7, 0x73, 0x1b,
};

//
// dump of bundle_requester.certchain.der
//
GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8  mTestBundleCert[] = {
  0x30, 0x82, 0x05, 0x19, 0x30, 0x82, 0x03, 0x01, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x14, 0x1d,
  0xaf, 0x4e, 0xf1, 0x99, 0xbf, 0xee, 0xe2, 0x6d, 0x7e, 0x4a, 0xac, 0xac, 0x20, 0x00, 0xeb, 0x78,
  0xf6, 0x6a, 0xe9, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b,
  0x05, 0x00, 0x30, 0x1c, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x11, 0x69,
  0x6e, 0x74, 0x65, 0x6c, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x43, 0x41,
  0x30, 0x1e, 0x17, 0x0d, 0x32, 0x32, 0x30, 0x31, 0x30, 0x35, 0x30, 0x36, 0x31, 0x31, 0x30, 0x35,
  0x5a, 0x17, 0x0d, 0x33, 0x32, 0x30, 0x31, 0x30, 0x33, 0x30, 0x36, 0x31, 0x31, 0x30, 0x35, 0x5a,
  0x30, 0x1c, 0x31, 0x1a, 0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x11, 0x69, 0x6e, 0x74,
  0x65, 0x6c, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x43, 0x41, 0x30, 0x82,
  0x02, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05,
  0x00, 0x03, 0x82, 0x02, 0x0f, 0x00, 0x30, 0x82, 0x02, 0x0a, 0x02, 0x82, 0x02, 0x01, 0x00, 0xd5,
  0x13, 0x55, 0xb1, 0x0d, 0x44, 0x17, 0x6a, 0x06, 0x39, 0x66, 0xee, 0xbc, 0x43, 0x81, 0xea, 0xe0,
  0x93, 0x3f, 0x70, 0xb5, 0x5b, 0x07, 0xde, 0xa0, 0x24, 0x47, 0xd0, 0xf8, 0x99, 0xe6, 0x50, 0x17,
  0xb4, 0x02, 0x60, 0xd9, 0x1a, 0x6c, 0xd3, 0x55, 0x60, 0x61, 0xde, 0x80, 0x6d, 0x38, 0x29, 0x46,
  0xbb, 0x45, 0xa2, 0xeb, 0x24, 0x1c, 0xbb, 0x87, 0x1f, 0x14, 0xed, 0x3a, 0xd5, 0x32, 0xd5, 0x11,
  0x4b, 0xf4, 0x99, 0x78, 0x00, 0x11, 0x8d, 0x6d, 0x6e, 0x5a, 0x0b, 0x81, 0xff, 0x08, 0x59, 0x01,
  0x2b, 0xf0, 0x9b, 0x52, 0x88, 0x41, 0xb6, 0x3e, 0x62, 0x7b, 0xbd, 0x25, 0x96, 0x64, 0x8e, 0xb1,
  0x2b, 0x6a, 0xf8, 0x39, 0x93, 0x0a, 0x6b, 0x3f, 0x8c, 0x2b, 0x35, 0x6e, 0x80, 0x32, 0x86, 0xf6,
  0x0f, 0x1e, 0x8c, 0xd4, 0x4e, 0x77, 0xbe, 0xaf, 0xe5, 0x4c, 0x47, 0x66, 0xdd, 0x62, 0xce, 0x64,
  0x65, 0x80, 0xca, 0xbd, 0x87, 0x94, 0x3e, 0xff, 0x3d, 0x0a, 0xd4, 0x39, 0xcc, 0xd1, 0xef, 0xd1,
  0x1a, 0x55, 0x2c, 0xeb, 0xdc, 0xcf, 0x9a, 0x59, 0x92, 0xda, 0xca, 0xf6, 0xae, 0x88, 0xbc, 0xc0,
  0x43, 0xe5, 0xa9, 0x60, 0xb2, 0x0f, 0x69, 0xa2, 0x2a, 0x98, 0x56, 0x0f, 0x4b, 0x0b, 0x63, 0xb3,
  0x4e, 0x8b, 0x43, 0xb4, 0x7f, 0x52, 0x1e, 0x5a, 0x9e, 0xdf, 0xa7, 0x98, 0x4d, 0xb9, 0x3b, 0x58,
  0x80, 0x69, 0x38, 0xb7, 0xeb, 0x20, 0x91, 0xbf, 0x59, 0x81, 0x44, 0xfc, 0xc6, 0xaf, 0xc2, 0xc8,
  0xe1, 0xae, 0x3d, 0xc5, 0xb6, 0x55, 0xa8, 0xa2, 0xcd, 0xec, 0x0b, 0xbe, 0x9c, 0x34, 0x2b, 0xf5,
  0x4b, 0x2e, 0x3b, 0x26, 0xaa, 0x22, 0x12, 0x29, 0xc2, 0x66, 0xc2, 0x40, 0xdd, 0x12, 0xfa, 0xf0,
  0xb4, 0xe8, 0x95, 0x20, 0x59, 0x91, 0x9e, 0x7d, 0x36, 0x6e, 0x2d, 0x99, 0x53, 0x44, 0x2b, 0x43,
  0x12, 0x08, 0x74, 0x76, 0xd5, 0x21, 0xd3, 0x08, 0x37, 0xe1, 0xe6, 0xe3, 0xc4, 0x67, 0x88, 0x29,
  0xea, 0x06, 0x97, 0x9f, 0x61, 0x0d, 0x5d, 0x39, 0x17, 0x03, 0x87, 0x89, 0x35, 0xe0, 0xd5, 0x87,
  0xa1, 0x5b, 0x0c, 0x3c, 0x35, 0x3d, 0xd6, 0x6f, 0xdd, 0x6f, 0xa5, 0x11, 0x9f, 0x80, 0x20, 0xca,
  0x8b, 0x67, 0xd2, 0x2c, 0x7c, 0x51, 0x68, 0x15, 0x2c, 0x9c, 0x08, 0x86, 0xc8, 0x6c, 0xf0, 0x9d,
  0xc2, 0x98, 0xb9, 0x6e, 0x31, 0xaf, 0x2b, 0x34, 0x9c, 0xe3, 0xd0, 0xe9, 0x85, 0x37, 0x2b, 0x1f,
  0x21, 0xd2, 0xb9, 0xa4, 0x8d, 0x3f, 0xa3, 0x1a, 0x9f, 0x7d, 0x1b, 0xb6, 0xe0, 0xf6, 0x19, 0xf4,
  0x19, 0xf8, 0x50, 0xa7, 0x1b, 0xec, 0xb9, 0xbb, 0xa4, 0xd2, 0x20, 0xf3, 0xed, 0xc7, 0xa7, 0x94,
  0x2b, 0x0b, 0x9a, 0x74, 0xb8, 0xba, 0xd7, 0x42, 0x97, 0xab, 0xb4, 0xe9, 0xbe, 0x1e, 0x07, 0x0c,
  0x05, 0x55, 0x54, 0xba, 0x05, 0x73, 0x99, 0xf2, 0x29, 0x88, 0x69, 0x4f, 0x1b, 0x2e, 0x87, 0x7b,
  0x74, 0x94, 0xae, 0xc2, 0x2f, 0x3c, 0x90, 0xb5, 0x5c, 0x01, 0x46, 0xc6, 0x42, 0xfb, 0xa9, 0xad,
  0x82, 0x1b, 0xd9, 0x38, 0xbe, 0x0d, 0x72, 0x4c, 0x06, 0x03, 0x92, 0xc2, 0xc8, 0x7f, 0x7b, 0xd4,
  0x86, 0x83, 0xb8, 0x62, 0xd9, 0xf7, 0x2f, 0xc1, 0x57, 0x88, 0x85, 0xa5, 0x61, 0xc5, 0x93, 0xfc,
  0x17, 0xe0, 0x98, 0x11, 0x88, 0x0d, 0x03, 0x0b, 0x6e, 0x01, 0x16, 0xcc, 0xf2, 0x3c, 0x5f, 0x61,
  0x8c, 0x27, 0x31, 0xca, 0x4d, 0x35, 0x7a, 0xc1, 0xf3, 0xfe, 0x06, 0xff, 0xca, 0x7b, 0x96, 0xad,
  0x0d, 0xff, 0x9e, 0x12, 0x44, 0x51, 0x6e, 0xf4, 0xaa, 0x2e, 0xce, 0xe8, 0xc7, 0xa6, 0xe9, 0xae,
  0xc0, 0xbe, 0xde, 0x7b, 0x93, 0x3d, 0x1d, 0xf9, 0xfb, 0x4d, 0xa7, 0x0b, 0x73, 0xfb, 0x7b, 0x02,
  0x03, 0x01, 0x00, 0x01, 0xa3, 0x53, 0x30, 0x51, 0x30, 0x1d, 0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04,
  0x16, 0x04, 0x14, 0x03, 0xfe, 0xb0, 0x3f, 0x09, 0xe5, 0xe1, 0xf4, 0x2e, 0x87, 0xfb, 0x41, 0x86,
  0xf6, 0x76, 0xd3, 0x93, 0xbe, 0x96, 0x85, 0x30, 0x1f, 0x06, 0x03, 0x55, 0x1d, 0x23, 0x04, 0x18,
  0x30, 0x16, 0x80, 0x14, 0x03, 0xfe, 0xb0, 0x3f, 0x09, 0xe5, 0xe1, 0xf4, 0x2e, 0x87, 0xfb, 0x41,
  0x86, 0xf6, 0x76, 0xd3, 0x93, 0xbe, 0x96, 0x85, 0x30, 0x0f, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
  0x01, 0xff, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48,
  0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01, 0x00, 0x02, 0x15, 0x73,
  0x60, 0x10, 0xbc, 0x95, 0xac, 0x63, 0x15, 0x09, 0x6b, 0x0c, 0x09, 0x4a, 0x74, 0xcd, 0xe3, 0x58,
  0xcc, 0x6c, 0xf3, 0x55, 0x80, 0xe9, 0x75, 0x47, 0x2c, 0x80, 0x4e, 0x5f, 0xfe, 0xbd, 0x2e, 0xd7,
  0x42, 0x89, 0x44, 0x57, 0xbe, 0x51, 0x84, 0x53, 0xb6, 0x40, 0x80, 0xd6, 0xcc, 0xe2, 0x80, 0x22,
  0x53, 0xd9, 0x31, 0x65, 0xf9, 0x3c, 0x8b, 0x7b, 0xe6, 0xa6, 0x6e, 0xfd, 0x9d, 0x27, 0xe5, 0xcd,
  0xfd, 0x82, 0xcf, 0xd6, 0x18, 0xbe, 0xa6, 0xed, 0x72, 0xce, 0x5f, 0x45, 0x12, 0x6a, 0xaa, 0x95,
  0x42, 0x25, 0x28, 0x8b, 0xfc, 0x4b, 0xc9, 0xad, 0xdd, 0xdb, 0x4c, 0x74, 0x10, 0x2e, 0x90, 0x2a,
  0x06, 0x4f, 0x2e, 0xb4, 0x54, 0xb9, 0xc0, 0x60, 0xb9, 0x4d, 0xee, 0x59, 0x1c, 0x18, 0x8d, 0xd1,
  0x49, 0xc3, 0xe9, 0x1b, 0xf1, 0xfc, 0xc3, 0x83, 0x1f, 0x6a, 0xb0, 0xfc, 0x8d, 0xfc, 0x30, 0xed,
  0x9c, 0xcb, 0x78, 0x52, 0xe8, 0x09, 0x3d, 0x4f, 0xdc, 0xbc, 0xad, 0x84, 0xc1, 0xd5, 0x5a, 0x0a,
  0x07, 0xa3, 0xf6, 0x42, 0xd7, 0x54, 0x55, 0x01, 0x8e, 0x53, 0xce, 0xcb, 0x2a, 0x11, 0xf7, 0x89,
  0x7e, 0xaf, 0x6f, 0x4c, 0xb9, 0x56, 0x4a, 0x67, 0x4a, 0xf9, 0x4f, 0x64, 0x15, 0xfa, 0xb0, 0xf9,
  0x97, 0xe2, 0xf6, 0xa8, 0xf4, 0xe7, 0x0a, 0x7a, 0x83, 0x4e, 0xf6, 0xe9, 0xac, 0x5e, 0xd9, 0xa8,
  0xea, 0x6b, 0x06, 0xcb, 0x2c, 0x41, 0xc1, 0x7e, 0xf5, 0x79, 0xfc, 0x7c, 0x05, 0x06, 0x8f, 0x27,
  0xaa, 0x3b, 0x61, 0x82, 0x72, 0x55, 0xa9, 0xa0, 0xa0, 0xa5, 0x69, 0x2f, 0x95, 0x40, 0xfc, 0xfe,
  0x4a, 0x0f, 0x7d, 0x8c, 0x89, 0xaa, 0xc0, 0x1d, 0x87, 0x03, 0xa1, 0xce, 0xee, 0x23, 0x4f, 0xc5,
  0x7c, 0xb4, 0xb6, 0x2b, 0x6f, 0x05, 0x30, 0xc9, 0x16, 0x51, 0xdf, 0xc7, 0x16, 0x3c, 0x08, 0x38,
  0x20, 0xf9, 0xc5, 0xe0, 0x4a, 0xfa, 0xcb, 0x8c, 0xc3, 0xc5, 0xbb, 0x5c, 0xad, 0xca, 0xc2, 0x52,
  0x45, 0x2f, 0x54, 0x70, 0x78, 0x33, 0x70, 0xc2, 0xed, 0x68, 0xf1, 0x89, 0x67, 0xa3, 0x19, 0x24,
  0xcb, 0x8f, 0x99, 0x1b, 0x28, 0x81, 0x6c, 0x4e, 0x25, 0xb1, 0x27, 0x3d, 0x9f, 0xe7, 0x3d, 0xa7,
  0x73, 0x9e, 0x4c, 0x1a, 0x63, 0x8e, 0xf9, 0xa7, 0xb6, 0x21, 0xe7, 0x4c, 0xdf, 0xfb, 0x36, 0x47,
  0xda, 0x2d, 0xbb, 0x52, 0x55, 0xf8, 0x44, 0x0d, 0x0c, 0xde, 0xe2, 0x13, 0x42, 0x1b, 0xa2, 0xad,
  0xa0, 0x0f, 0x39, 0x6c, 0x78, 0x32, 0x7a, 0x03, 0x9e, 0x55, 0x4e, 0x43, 0xf7, 0x0c, 0x35, 0xd9,
  0x1d, 0x2c, 0x0f, 0x30, 0x30, 0x3e, 0x09, 0xe2, 0x31, 0xa6, 0xb0, 0x1e, 0xa9, 0xf5, 0x4b, 0xa1,
  0x74, 0x09, 0x50, 0xd4, 0xd3, 0xd3, 0x3e, 0x76, 0xb1, 0x67, 0xfc, 0x51, 0xb0, 0x93, 0x22, 0xc4,
  0x6b, 0x8a, 0x27, 0x45, 0x19, 0x3b, 0x35, 0x91, 0x61, 0x36, 0xe7, 0x9c, 0xf6, 0xda, 0x96, 0x30,
  0x7d, 0xf4, 0x11, 0xc4, 0x3f, 0x35, 0x4e, 0x7a, 0xd6, 0x6e, 0xb6, 0xea, 0xe7, 0x66, 0x6f, 0x23,
  0x5c, 0x53, 0x76, 0x53, 0xfc, 0x35, 0x93, 0xe5, 0xfc, 0xb9, 0x6b, 0xb9, 0xd3, 0x6c, 0x48, 0x66,
  0x6f, 0xd7, 0x10, 0x6e, 0x25, 0x72, 0x71, 0x31, 0xfa, 0xc0, 0xdf, 0x31, 0xca, 0xd1, 0xaf, 0xc2,
  0x8e, 0xa5, 0xca, 0xd7, 0x3f, 0x4e, 0xde, 0x47, 0xd5, 0x8e, 0xfc, 0x75, 0xb6, 0x71, 0x83, 0xd9,
  0xfd, 0x11, 0x35, 0x81, 0xbf, 0x10, 0x0d, 0x3e, 0x50, 0x45, 0x07, 0x39, 0x08, 0x73, 0x7a, 0x0b,
  0x21, 0x32, 0xaf, 0xf4, 0x99, 0xeb, 0x4d, 0xd4, 0xe8, 0x2a, 0x06, 0x98, 0x43, 0xbb, 0xbb, 0x11,
  0x63, 0x99, 0xa8, 0x41, 0x22, 0xe8, 0x86, 0x79, 0x4b, 0x53, 0xb7, 0x73, 0x1b, 0x30, 0x82, 0x04,
  0xa0, 0x30, 0x82, 0x02, 0x88, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x01, 0x30, 0x0d, 0x06,
  0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30, 0x1c, 0x31, 0x1a,
  0x30, 0x18, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x11, 0x69, 0x6e, 0x74, 0x65, 0x6c, 0x20, 0x74,
  0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x43, 0x41, 0x30, 0x1e, 0x17, 0x0d, 0x32, 0x32,
  0x30, 0x31, 0x30, 0x35, 0x30, 0x36, 0x31, 0x31, 0x30, 0x36, 0x5a, 0x17, 0x0d, 0x33, 0x32, 0x30,
  0x31, 0x30, 0x33, 0x30, 0x36, 0x31, 0x31, 0x30, 0x36, 0x5a, 0x30, 0x2b, 0x31, 0x29, 0x30, 0x27,
  0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x6c, 0x20, 0x74, 0x65, 0x73,
  0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61,
  0x74, 0x65, 0x20, 0x63, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0xa2, 0x30, 0x0d, 0x06, 0x09, 0x2a,
  0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x8f, 0x00, 0x30,
  0x82, 0x01, 0x8a, 0x02, 0x82, 0x01, 0x81, 0x00, 0xd1, 0xcf, 0x2d, 0x21, 0x38, 0xe4, 0x81, 0xa6,
  0xf2, 0xe5, 0xca, 0xac, 0x09, 0x2b, 0x04, 0x2a, 0x52, 0x76, 0x2f, 0xa3, 0x4b, 0x43, 0xc2, 0xd6,
  0xb5, 0x3e, 0x2b, 0x29, 0x6f, 0x1b, 0x7d, 0x36, 0x60, 0xcd, 0x71, 0xf4, 0xcf, 0x8a, 0x24, 0xac,
  0x8f, 0x5f, 0x78, 0x70, 0xc9, 0x84, 0x48, 0xfb, 0x1d, 0xcb, 0xee, 0x34, 0x3c, 0x4c, 0xe1, 0x1a,
  0x92, 0x54, 0x43, 0x45, 0xe0, 0xdd, 0x14, 0xd5, 0xac, 0x13, 0xae, 0xcc, 0x45, 0xdc, 0x52, 0xdd,
  0x45, 0x22, 0x10, 0xab, 0xb2, 0x1b, 0x3e, 0xad, 0x36, 0x6f, 0x00, 0x59, 0xb6, 0xfd, 0x27, 0xb8,
  0x57, 0xbb, 0x66, 0x72, 0x1e, 0x7b, 0xe8, 0xc7, 0xaf, 0x5e, 0x65, 0x5c, 0x58, 0x3d, 0x9e, 0x78,
  0x26, 0xb7, 0x70, 0x9a, 0xe7, 0x3c, 0xfc, 0x90, 0x64, 0xb2, 0xb2, 0xb2, 0x80, 0x22, 0xb3, 0x1c,
  0x23, 0x73, 0x6a, 0xe3, 0xcb, 0x5b, 0x4e, 0x29, 0x71, 0xdd, 0xfd, 0xd7, 0x15, 0x3a, 0xd6, 0xa3,
  0xe0, 0x4a, 0x53, 0x65, 0xa6, 0xae, 0x6a, 0x83, 0xb0, 0x00, 0x3f, 0x64, 0x4b, 0x01, 0xcd, 0x97,
  0xb1, 0xe1, 0x54, 0x1e, 0xa2, 0x33, 0x4e, 0xf0, 0x13, 0x33, 0x80, 0xe5, 0x41, 0x9c, 0x5e, 0xb3,
  0x5d, 0xae, 0x53, 0x2c, 0x41, 0x51, 0xf2, 0x00, 0x3e, 0x03, 0xe2, 0x0d, 0xa4, 0xf5, 0x24, 0x08,
  0x46, 0xc2, 0xed, 0x6b, 0xf2, 0xe0, 0x1b, 0x04, 0xb7, 0xe8, 0xda, 0x85, 0xa4, 0x2f, 0x4b, 0x53,
  0xfa, 0x76, 0x72, 0x0a, 0x28, 0xbf, 0x3c, 0xa8, 0x85, 0x03, 0xee, 0x51, 0xb5, 0x5c, 0xa9, 0xa5,
  0x70, 0x47, 0x07, 0x11, 0x32, 0xc5, 0x5f, 0x7a, 0x07, 0xfb, 0x5f, 0x4a, 0x52, 0x65, 0x33, 0x0c,
  0x17, 0x8e, 0x74, 0xe5, 0xec, 0x7d, 0x52, 0xa0, 0xd4, 0xef, 0x64, 0xcc, 0x5e, 0xe2, 0x3d, 0x46,
  0x1e, 0x73, 0xab, 0x43, 0x2d, 0x05, 0xfa, 0xa3, 0x6d, 0x86, 0x02, 0xe7, 0xc8, 0x5f, 0x44, 0xfe,
  0xeb, 0x14, 0x73, 0x2d, 0xf4, 0x0b, 0x72, 0x49, 0xe3, 0xef, 0x55, 0x05, 0x1b, 0x2f, 0xf9, 0x73,
  0xeb, 0xed, 0xcc, 0x5e, 0x52, 0x17, 0x43, 0x94, 0x85, 0xc7, 0x0e, 0x73, 0x89, 0x22, 0xdc, 0x02,
  0x5f, 0x99, 0x69, 0x58, 0x16, 0x89, 0x1e, 0x6f, 0x75, 0xf8, 0xd3, 0xe7, 0x0a, 0xb1, 0x01, 0x42,
  0x71, 0xa6, 0xc5, 0xb2, 0xd0, 0xc0, 0x75, 0x96, 0x2e, 0xc2, 0x32, 0x49, 0x0a, 0xe8, 0x85, 0xbd,
  0x22, 0x4c, 0x95, 0xd4, 0xf8, 0x2d, 0x72, 0x05, 0x4f, 0x0e, 0x56, 0x9c, 0xc3, 0x30, 0x3d, 0xfd,
  0x30, 0x63, 0x92, 0x72, 0x6b, 0xaf, 0x80, 0x9d, 0xd0, 0xc2, 0x36, 0xe7, 0xc1, 0x37, 0xd0, 0x64,
  0x88, 0x64, 0x3c, 0x33, 0xdd, 0x87, 0xdd, 0xd9, 0xa2, 0x79, 0x44, 0xd8, 0x25, 0x84, 0x0e, 0xfd,
  0x9d, 0xd6, 0xc1, 0x7a, 0x2d, 0x23, 0x08, 0x9d, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x5e, 0x30,
  0x5c, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x05, 0x30, 0x03, 0x01, 0x01, 0xff, 0x30,
  0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x01, 0xfe, 0x30, 0x1d, 0x06, 0x03,
  0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x46, 0xcd, 0xb4, 0x52, 0x02, 0x29, 0x26, 0xc7, 0x52,
  0x23, 0x81, 0x65, 0xcd, 0x87, 0x2f, 0x96, 0x1b, 0x01, 0x2c, 0xc5, 0x30, 0x20, 0x06, 0x03, 0x55,
  0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x16, 0x30, 0x14, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05,
  0x07, 0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x02, 0x30, 0x0d, 0x06,
  0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x02, 0x01,
  0x00, 0x71, 0x14, 0xca, 0xfe, 0x03, 0x92, 0xc6, 0x88, 0x52, 0x76, 0x43, 0xd9, 0x8a, 0xf7, 0x22,
  0x94, 0xef, 0x40, 0xa7, 0xc3, 0x7b, 0x6f, 0xb4, 0xd0, 0x42, 0xc3, 0x59, 0x1f, 0x77, 0xef, 0xac,
  0x18, 0xce, 0x7c, 0xab, 0x48, 0x85, 0x22, 0xf1, 0x50, 0x2c, 0xe0, 0x63, 0x4d, 0x9d, 0x0e, 0xed,
  0x38, 0xff, 0xd9, 0xed, 0x3c, 0x7e, 0x3e, 0x84, 0xb1, 0xa7, 0x78, 0x62, 0x37, 0x39, 0x6d, 0xd5,
  0x5d, 0x46, 0x9a, 0x6f, 0x63, 0xc2, 0xa7, 0x90, 0xf9, 0x7a, 0xa7, 0x4b, 0xa2, 0x8a, 0xa8, 0x50,
  0x13, 0x16, 0x9c, 0x4f, 0xcc, 0x28, 0x58, 0x6c, 0x9f, 0x65, 0x6e, 0xf0, 0x9a, 0x92, 0x59, 0x94,
  0xa4, 0xb5, 0xfd, 0x8f, 0xf0, 0x7c, 0x1e, 0xba, 0xc3, 0x21, 0x13, 0xfb, 0x33, 0xba, 0xad, 0x4b,
  0x1d, 0x42, 0x23, 0x16, 0x8b, 0x3e, 0xcd, 0xb5, 0x69, 0x9c, 0x7c, 0xa1, 0x9b, 0x92, 0xc1, 0x2f,
  0x9c, 0x89, 0x3f, 0x28, 0x24, 0x67, 0x48, 0x47, 0xf0, 0xb1, 0xc0, 0xf0, 0x4c, 0xbf, 0xf6, 0x64,
  0xde, 0x33, 0xc3, 0xd6, 0x5b, 0xe6, 0xc3, 0xdd, 0xa5, 0xba, 0x42, 0x02, 0x0f, 0xbc, 0xbe, 0xde,
  0x14, 0x12, 0x7c, 0xfb, 0xe6, 0xbb, 0xdd, 0xf2, 0x6b, 0xd7, 0x75, 0xe6, 0xf5, 0xda, 0xbe, 0xe9,
  0xbf, 0xa6, 0x87, 0x09, 0x41, 0x6c, 0x9d, 0x19, 0x9c, 0xb1, 0x7c, 0x89, 0x9f, 0x32, 0xb3, 0x0a,
  0x59, 0x10, 0xc6, 0x8c, 0xa0, 0x5b, 0xda, 0xa6, 0xa3, 0xf8, 0x33, 0x77, 0x7a, 0xf4, 0xb0, 0x8a,
  0xcf, 0xc8, 0x3f, 0x2e, 0xb8, 0xf1, 0x9e, 0x7b, 0x47, 0x2d, 0x85, 0x3b, 0x58, 0xcd, 0x4b, 0xbc,
  0x87, 0xd2, 0xf3, 0xcd, 0x36, 0x80, 0x9b, 0x4a, 0x0f, 0x24, 0x30, 0x1d, 0x3f, 0x6f, 0x56, 0xcf,
  0x6e, 0x2d, 0xcf, 0xa8, 0x17, 0xd5, 0x97, 0x0b, 0x22, 0xa6, 0x59, 0xef, 0xef, 0xd4, 0x9a, 0x0e,
  0x6f, 0xb9, 0xf0, 0x48, 0x2a, 0x54, 0x22, 0x77, 0x27, 0x90, 0x84, 0x42, 0x56, 0x85, 0x80, 0x78,
  0x4f, 0xd9, 0x14, 0x2a, 0xf0, 0x5d, 0x6b, 0x46, 0x60, 0x3d, 0xad, 0xa5, 0xa2, 0xb9, 0x04, 0x23,
  0x92, 0x1b, 0x70, 0xa3, 0xdb, 0xaa, 0xf0, 0x5f, 0x1c, 0xd8, 0x26, 0xbb, 0x51, 0x17, 0xfc, 0x93,
  0x6d, 0x70, 0x19, 0x54, 0xe2, 0x6f, 0x82, 0x8a, 0x49, 0xa6, 0x19, 0xcc, 0x97, 0x53, 0x90, 0x27,
  0xd9, 0x8d, 0xaa, 0x8c, 0xc3, 0x2b, 0xbc, 0x0a, 0x72, 0x3a, 0x41, 0xb8, 0xfa, 0x1e, 0xbb, 0x8b,
  0x27, 0x06, 0x75, 0x53, 0x91, 0xac, 0x8d, 0x75, 0xf2, 0xa6, 0xea, 0x85, 0x7e, 0x34, 0x06, 0x9e,
  0xf9, 0xe9, 0x13, 0xa3, 0xfe, 0x2e, 0x38, 0x4b, 0x3f, 0x61, 0x11, 0x1f, 0x6b, 0xd3, 0xfe, 0xc9,
  0x13, 0xbc, 0x12, 0xfe, 0xbc, 0xff, 0xaa, 0x38, 0x73, 0x8a, 0x73, 0x8c, 0xcf, 0xcc, 0xd3, 0xe7,
  0xba, 0x98, 0xfd, 0xf4, 0xf5, 0xf6, 0xa9, 0xc8, 0x07, 0x5b, 0x16, 0xae, 0x76, 0x86, 0x24, 0x25,
  0xcc, 0x7c, 0xdb, 0x1e, 0x7f, 0xc4, 0xe4, 0xd3, 0x89, 0x5a, 0x91, 0x6a, 0x9c, 0x93, 0x03, 0xd2,
  0xd2, 0xc0, 0x29, 0x69, 0x58, 0x52, 0x6c, 0xca, 0x91, 0x7d, 0xd2, 0x3d, 0xc7, 0x2f, 0x8a, 0x4e,
  0x55, 0x03, 0x34, 0x16, 0x01, 0x2f, 0x4f, 0x57, 0xea, 0x10, 0xa7, 0xfb, 0xe3, 0x2e, 0x2c, 0x7f,
  0x5d, 0x27, 0x93, 0x74, 0x95, 0x25, 0x1a, 0x1e, 0x54, 0x1a, 0xb6, 0xe1, 0xdc, 0xc8, 0xe4, 0x84,
  0xeb, 0x38, 0x0f, 0x05, 0x74, 0xf3, 0x3a, 0xf9, 0xd8, 0x00, 0x43, 0xe9, 0x24, 0x1e, 0x45, 0xb4,
  0xe2, 0x38, 0x8b, 0x6a, 0x4f, 0xb6, 0x0b, 0x1d, 0xac, 0xbc, 0xec, 0xda, 0x41, 0x9c, 0x7f, 0xa8,
  0x15, 0x09, 0x3b, 0x0b, 0x99, 0xc4, 0x48, 0xdd, 0xde, 0x0d, 0x65, 0x86, 0x72, 0xaa, 0x0a, 0x4e,
  0x71, 0x30, 0x82, 0x03, 0xeb, 0x30, 0x82, 0x02, 0x53, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01,
  0x02, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00,
  0x30, 0x2b, 0x31, 0x29, 0x30, 0x27, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x20, 0x69, 0x6e, 0x74,
  0x65, 0x6c, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x69, 0x6e, 0x74, 0x65,
  0x72, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x63, 0x65, 0x72, 0x74, 0x30, 0x1e, 0x17,
  0x0d, 0x32, 0x32, 0x30, 0x31, 0x30, 0x35, 0x30, 0x36, 0x31, 0x31, 0x30, 0x36, 0x5a, 0x17, 0x0d,
  0x33, 0x32, 0x30, 0x31, 0x30, 0x33, 0x30, 0x36, 0x31, 0x31, 0x30, 0x36, 0x5a, 0x30, 0x28, 0x31,
  0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x1d, 0x69, 0x6e, 0x74, 0x65, 0x6c, 0x20,
  0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x72, 0x65, 0x71, 0x75, 0x73, 0x65, 0x74,
  0x65, 0x72, 0x20, 0x63, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a,
  0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30,
  0x82, 0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0x93, 0x47, 0xf3, 0x71, 0x6b, 0x24, 0x1e, 0xc0,
  0xeb, 0x6b, 0x22, 0x23, 0xe7, 0xc8, 0x10, 0x9d, 0xb2, 0xef, 0x1e, 0x65, 0xe9, 0xff, 0xd6, 0x37,
  0xee, 0xc5, 0x3f, 0x29, 0x25, 0x94, 0xb4, 0x30, 0x97, 0x51, 0xe3, 0x51, 0xa4, 0x39, 0xd1, 0x27,
  0x32, 0x9e, 0x6b, 0x80, 0xc2, 0x29, 0xf2, 0x28, 0xd1, 0x49, 0xdc, 0x62, 0x27, 0x5b, 0x3f, 0xce,
  0x83, 0xd7, 0x7b, 0x09, 0x37, 0x48, 0x16, 0x7f, 0x2e, 0x6e, 0xfa, 0x67, 0xaa, 0x92, 0x17, 0xcb,
  0xff, 0x4b, 0xe3, 0x1b, 0xd1, 0xa6, 0xe3, 0x6c, 0x4d, 0x9f, 0x9e, 0xc7, 0x01, 0xed, 0x9f, 0x14,
  0xb0, 0x34, 0x44, 0xa2, 0x88, 0x28, 0xf2, 0x3f, 0xad, 0xf2, 0xf7, 0x51, 0x8c, 0xfe, 0x43, 0x73,
  0xfa, 0x8d, 0x2f, 0x63, 0x8c, 0x0e, 0xe5, 0x27, 0xdc, 0x12, 0x81, 0x26, 0xea, 0x92, 0x67, 0x7d,
  0xc9, 0xfc, 0xec, 0x4c, 0xcc, 0x79, 0x4d, 0x2d, 0xfe, 0xf6, 0x63, 0xb7, 0x63, 0xca, 0x70, 0x24,
  0xe0, 0x23, 0x53, 0x92, 0xe8, 0x56, 0xb7, 0x85, 0x6d, 0x25, 0x9e, 0xe0, 0x24, 0xa8, 0x5c, 0xe0,
  0x0f, 0xc1, 0xb6, 0x20, 0x2f, 0x85, 0x2a, 0x67, 0xf6, 0x1b, 0x58, 0x60, 0x5a, 0x14, 0xda, 0xc2,
  0x03, 0x10, 0x79, 0x33, 0x3c, 0x41, 0xc6, 0xbe, 0xd2, 0xee, 0x2f, 0x65, 0xd5, 0xad, 0x9c, 0xc6,
  0x09, 0xae, 0x26, 0xf2, 0xac, 0xc2, 0x65, 0x12, 0x74, 0x09, 0xe8, 0x89, 0x66, 0xf6, 0x95, 0xb8,
  0x6a, 0x5f, 0x96, 0xc2, 0x3c, 0x9f, 0x01, 0x52, 0xa8, 0xc8, 0x4e, 0xd8, 0xba, 0x95, 0x38, 0x5b,
  0xf8, 0xc6, 0x43, 0x54, 0xac, 0x63, 0x90, 0xd4, 0xde, 0x11, 0x40, 0x27, 0xe5, 0x12, 0x1d, 0x72,
  0xa2, 0xec, 0xad, 0x0a, 0x8b, 0x68, 0x21, 0x9d, 0xea, 0x16, 0x70, 0x5f, 0x32, 0x3a, 0xed, 0x4f,
  0x0b, 0xb2, 0x44, 0x1f, 0x44, 0x9b, 0x4c, 0x03, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0x9c,
  0x30, 0x81, 0x99, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30,
  0x00, 0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x05, 0xe0, 0x30, 0x1d,
  0x06, 0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x7c, 0x69, 0x5e, 0x84, 0xbb, 0xc7, 0x6b,
  0xfc, 0x45, 0x70, 0xa4, 0x4b, 0x1e, 0x67, 0x70, 0x04, 0xa4, 0x37, 0xdd, 0x72, 0x30, 0x31, 0x06,
  0x03, 0x55, 0x1d, 0x11, 0x04, 0x2a, 0x30, 0x28, 0xa0, 0x26, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04,
  0x01, 0x83, 0x1c, 0x82, 0x12, 0x01, 0xa0, 0x18, 0x0c, 0x16, 0x41, 0x43, 0x4d, 0x45, 0x3a, 0x57,
  0x49, 0x44, 0x47, 0x45, 0x54, 0x3a, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30,
  0x30, 0x2a, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x20, 0x30, 0x1e, 0x06, 0x08,
  0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07,
  0x03, 0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x09, 0x30, 0x0d, 0x06, 0x09,
  0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x81, 0x00,
  0x8e, 0x19, 0x08, 0xec, 0x7a, 0xcc, 0x6d, 0x04, 0x5f, 0xb7, 0xf9, 0xd2, 0xcd, 0x76, 0xee, 0x44,
  0x1b, 0x7c, 0x19, 0xcb, 0x18, 0xdd, 0x60, 0x50, 0x3a, 0x54, 0xb1, 0x75, 0x17, 0xf6, 0xce, 0x19,
  0x15, 0x09, 0x9e, 0x5b, 0xfc, 0x5c, 0xb3, 0x20, 0x8d, 0xcb, 0x9d, 0xfc, 0x23, 0x12, 0x68, 0x87,
  0x55, 0x0d, 0x3b, 0x5f, 0x9f, 0x4e, 0xb2, 0x17, 0x13, 0x0b, 0xc4, 0x6f, 0xbe, 0x43, 0x75, 0xa6,
  0xb7, 0x3c, 0x0e, 0xe0, 0xdf, 0x84, 0xe5, 0x88, 0x46, 0x08, 0xc3, 0x36, 0x1b, 0x31, 0x02, 0x92,
  0x7c, 0x53, 0xc4, 0x08, 0x63, 0x3e, 0x46, 0x75, 0xd7, 0x35, 0x9a, 0xd0, 0x76, 0x71, 0xf9, 0x57,
  0x97, 0xc7, 0x3c, 0x3b, 0xce, 0x7a, 0x82, 0x95, 0x15, 0x8e, 0x20, 0xcc, 0x7b, 0xa0, 0xc4, 0x68,
  0x21, 0x26, 0x9c, 0xfd, 0x29, 0x83, 0x41, 0x19, 0x98, 0xb6, 0x8a, 0x3a, 0x06, 0x5f, 0x01, 0x1b,
  0x80, 0xac, 0x33, 0xd9, 0x0c, 0x9c, 0xea, 0x70, 0xd7, 0xf5, 0x1e, 0xb0, 0x78, 0x24, 0xbc, 0x59,
  0xaf, 0x07, 0xc6, 0x16, 0x46, 0xdd, 0x9d, 0x00, 0x9a, 0xc8, 0x9a, 0x04, 0x19, 0x4d, 0x62, 0xa9,
  0x4f, 0x7c, 0x05, 0x24, 0x93, 0xc6, 0x01, 0xc5, 0xb2, 0x89, 0xe8, 0x62, 0x47, 0xbe, 0xa3, 0xd1,
  0x9c, 0x47, 0x31, 0x06, 0x16, 0x6b, 0x3b, 0xf8, 0xad, 0xb0, 0x4c, 0xfb, 0x2b, 0x6b, 0x8e, 0x88,
  0x53, 0x70, 0x75, 0x40, 0x30, 0xaf, 0xfe, 0xc5, 0xf3, 0x0f, 0x86, 0x8f, 0x58, 0x56, 0x67, 0xbd,
  0x15, 0x1f, 0x8a, 0x5c, 0xa3, 0x8a, 0x9a, 0x88, 0xe5, 0xf2, 0xd3, 0x7f, 0xac, 0x56, 0x34, 0x21,
  0x24, 0xf7, 0xde, 0x9d, 0x97, 0x9b, 0xe6, 0x20, 0xd6, 0x3d, 0x48, 0x73, 0x29, 0x23, 0xf9, 0x0f,
  0xab, 0x04, 0xe6, 0x1d, 0x70, 0xee, 0xcb, 0x5a, 0xe5, 0x8a, 0xf4, 0xbc, 0x4b, 0xad, 0x18, 0xec,
  0x25, 0x8a, 0xb8, 0x91, 0xf9, 0x53, 0xf4, 0xe4, 0xce, 0xa7, 0x54, 0xf6, 0x83, 0x66, 0x07, 0xc5,
  0x51, 0x48, 0x20, 0x54, 0x5b, 0x5d, 0xc1, 0x80, 0x04, 0x17, 0x89, 0x14, 0x5a, 0x06, 0x34, 0x86,
  0xdb, 0x8b, 0xaf, 0x04, 0x24, 0xd9, 0xbd, 0xa8, 0x27, 0x09, 0x51, 0xfe, 0x6e, 0x47, 0x3c, 0x29,
  0x0a, 0x64, 0x3e, 0x0f, 0x00, 0xf1, 0xd9, 0xdc, 0xe9, 0xd1, 0x3c, 0xcf, 0xa4, 0xa9, 0x4f, 0x54,
  0x02, 0xcf, 0xa1, 0x40, 0x49, 0x61, 0x2a, 0x07, 0xa2, 0x44, 0x35, 0x20, 0x20, 0x56, 0x1c, 0x27,
  0xa2, 0xff, 0x07, 0x3e, 0x8b, 0xfe, 0x60, 0x42, 0x75, 0x0c, 0xe3, 0x8f, 0xee, 0x33, 0x8d, 0xfc,
  0x3b, 0x53, 0x36, 0x39, 0x10, 0x1a, 0xe3, 0xd2, 0x09, 0x7b, 0x00, 0xd0, 0xbd, 0x39, 0xf4, 0xcc,
  0xe3, 0x42, 0xd4, 0xa5, 0x2d, 0x0b, 0x58, 0xdf, 0x24, 0x7f, 0x85, 0x0a, 0xf2, 0x17, 0x76, 0xdd,
};

//
// dump of end_requester.key.der
//
GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8  mTestEndCert[] = {
  0x30, 0x82, 0x03, 0xeb, 0x30, 0x82, 0x02, 0x53, 0xa0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x01, 0x02,
  0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x30,
  0x2b, 0x31, 0x29, 0x30, 0x27, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x20, 0x69, 0x6e, 0x74, 0x65,
  0x6c, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72,
  0x6d, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x63, 0x65, 0x72, 0x74, 0x30, 0x1e, 0x17, 0x0d,
  0x32, 0x32, 0x30, 0x31, 0x30, 0x35, 0x30, 0x36, 0x31, 0x31, 0x30, 0x36, 0x5a, 0x17, 0x0d, 0x33,
  0x32, 0x30, 0x31, 0x30, 0x33, 0x30, 0x36, 0x31, 0x31, 0x30, 0x36, 0x5a, 0x30, 0x28, 0x31, 0x26,
  0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x0c, 0x1d, 0x69, 0x6e, 0x74, 0x65, 0x6c, 0x20, 0x74,
  0x65, 0x73, 0x74, 0x20, 0x52, 0x53, 0x41, 0x20, 0x72, 0x65, 0x71, 0x75, 0x73, 0x65, 0x74, 0x65,
  0x72, 0x20, 0x63, 0x65, 0x72, 0x74, 0x30, 0x82, 0x01, 0x22, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86,
  0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0f, 0x00, 0x30, 0x82,
  0x01, 0x0a, 0x02, 0x82, 0x01, 0x01, 0x00, 0x93, 0x47, 0xf3, 0x71, 0x6b, 0x24, 0x1e, 0xc0, 0xeb,
  0x6b, 0x22, 0x23, 0xe7, 0xc8, 0x10, 0x9d, 0xb2, 0xef, 0x1e, 0x65, 0xe9, 0xff, 0xd6, 0x37, 0xee,
  0xc5, 0x3f, 0x29, 0x25, 0x94, 0xb4, 0x30, 0x97, 0x51, 0xe3, 0x51, 0xa4, 0x39, 0xd1, 0x27, 0x32,
  0x9e, 0x6b, 0x80, 0xc2, 0x29, 0xf2, 0x28, 0xd1, 0x49, 0xdc, 0x62, 0x27, 0x5b, 0x3f, 0xce, 0x83,
  0xd7, 0x7b, 0x09, 0x37, 0x48, 0x16, 0x7f, 0x2e, 0x6e, 0xfa, 0x67, 0xaa, 0x92, 0x17, 0xcb, 0xff,
  0x4b, 0xe3, 0x1b, 0xd1, 0xa6, 0xe3, 0x6c, 0x4d, 0x9f, 0x9e, 0xc7, 0x01, 0xed, 0x9f, 0x14, 0xb0,
  0x34, 0x44, 0xa2, 0x88, 0x28, 0xf2, 0x3f, 0xad, 0xf2, 0xf7, 0x51, 0x8c, 0xfe, 0x43, 0x73, 0xfa,
  0x8d, 0x2f, 0x63, 0x8c, 0x0e, 0xe5, 0x27, 0xdc, 0x12, 0x81, 0x26, 0xea, 0x92, 0x67, 0x7d, 0xc9,
  0xfc, 0xec, 0x4c, 0xcc, 0x79, 0x4d, 0x2d, 0xfe, 0xf6, 0x63, 0xb7, 0x63, 0xca, 0x70, 0x24, 0xe0,
  0x23, 0x53, 0x92, 0xe8, 0x56, 0xb7, 0x85, 0x6d, 0x25, 0x9e, 0xe0, 0x24, 0xa8, 0x5c, 0xe0, 0x0f,
  0xc1, 0xb6, 0x20, 0x2f, 0x85, 0x2a, 0x67, 0xf6, 0x1b, 0x58, 0x60, 0x5a, 0x14, 0xda, 0xc2, 0x03,
  0x10, 0x79, 0x33, 0x3c, 0x41, 0xc6, 0xbe, 0xd2, 0xee, 0x2f, 0x65, 0xd5, 0xad, 0x9c, 0xc6, 0x09,
  0xae, 0x26, 0xf2, 0xac, 0xc2, 0x65, 0x12, 0x74, 0x09, 0xe8, 0x89, 0x66, 0xf6, 0x95, 0xb8, 0x6a,
  0x5f, 0x96, 0xc2, 0x3c, 0x9f, 0x01, 0x52, 0xa8, 0xc8, 0x4e, 0xd8, 0xba, 0x95, 0x38, 0x5b, 0xf8,
  0xc6, 0x43, 0x54, 0xac, 0x63, 0x90, 0xd4, 0xde, 0x11, 0x40, 0x27, 0xe5, 0x12, 0x1d, 0x72, 0xa2,
  0xec, 0xad, 0x0a, 0x8b, 0x68, 0x21, 0x9d, 0xea, 0x16, 0x70, 0x5f, 0x32, 0x3a, 0xed, 0x4f, 0x0b,
  0xb2, 0x44, 0x1f, 0x44, 0x9b, 0x4c, 0x03, 0x02, 0x03, 0x01, 0x00, 0x01, 0xa3, 0x81, 0x9c, 0x30,
  0x81, 0x99, 0x30, 0x0c, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x02, 0x30, 0x00,
  0x30, 0x0b, 0x06, 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x05, 0xe0, 0x30, 0x1d, 0x06,
  0x03, 0x55, 0x1d, 0x0e, 0x04, 0x16, 0x04, 0x14, 0x7c, 0x69, 0x5e, 0x84, 0xbb, 0xc7, 0x6b, 0xfc,
  0x45, 0x70, 0xa4, 0x4b, 0x1e, 0x67, 0x70, 0x04, 0xa4, 0x37, 0xdd, 0x72, 0x30, 0x31, 0x06, 0x03,
  0x55, 0x1d, 0x11, 0x04, 0x2a, 0x30, 0x28, 0xa0, 0x26, 0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01,
  0x83, 0x1c, 0x82, 0x12, 0x01, 0xa0, 0x18, 0x0c, 0x16, 0x41, 0x43, 0x4d, 0x45, 0x3a, 0x57, 0x49,
  0x44, 0x47, 0x45, 0x54, 0x3a, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x30,
  0x2a, 0x06, 0x03, 0x55, 0x1d, 0x25, 0x01, 0x01, 0xff, 0x04, 0x20, 0x30, 0x1e, 0x06, 0x08, 0x2b,
  0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03,
  0x02, 0x06, 0x08, 0x2b, 0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x09, 0x30, 0x0d, 0x06, 0x09, 0x2a,
  0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0b, 0x05, 0x00, 0x03, 0x82, 0x01, 0x81, 0x00, 0x8e,
  0x19, 0x08, 0xec, 0x7a, 0xcc, 0x6d, 0x04, 0x5f, 0xb7, 0xf9, 0xd2, 0xcd, 0x76, 0xee, 0x44, 0x1b,
  0x7c, 0x19, 0xcb, 0x18, 0xdd, 0x60, 0x50, 0x3a, 0x54, 0xb1, 0x75, 0x17, 0xf6, 0xce, 0x19, 0x15,
  0x09, 0x9e, 0x5b, 0xfc, 0x5c, 0xb3, 0x20, 0x8d, 0xcb, 0x9d, 0xfc, 0x23, 0x12, 0x68, 0x87, 0x55,
  0x0d, 0x3b, 0x5f, 0x9f, 0x4e, 0xb2, 0x17, 0x13, 0x0b, 0xc4, 0x6f, 0xbe, 0x43, 0x75, 0xa6, 0xb7,
  0x3c, 0x0e, 0xe0, 0xdf, 0x84, 0xe5, 0x88, 0x46, 0x08, 0xc3, 0x36, 0x1b, 0x31, 0x02, 0x92, 0x7c,
  0x53, 0xc4, 0x08, 0x63, 0x3e, 0x46, 0x75, 0xd7, 0x35, 0x9a, 0xd0, 0x76, 0x71, 0xf9, 0x57, 0x97,
  0xc7, 0x3c, 0x3b, 0xce, 0x7a, 0x82, 0x95, 0x15, 0x8e, 0x20, 0xcc, 0x7b, 0xa0, 0xc4, 0x68, 0x21,
  0x26, 0x9c, 0xfd, 0x29, 0x83, 0x41, 0x19, 0x98, 0xb6, 0x8a, 0x3a, 0x06, 0x5f, 0x01, 0x1b, 0x80,
  0xac, 0x33, 0xd9, 0x0c, 0x9c, 0xea, 0x70, 0xd7, 0xf5, 0x1e, 0xb0, 0x78, 0x24, 0xbc, 0x59, 0xaf,
  0x07, 0xc6, 0x16, 0x46, 0xdd, 0x9d, 0x00, 0x9a, 0xc8, 0x9a, 0x04, 0x19, 0x4d, 0x62, 0xa9, 0x4f,
  0x7c, 0x05, 0x24, 0x93, 0xc6, 0x01, 0xc5, 0xb2, 0x89, 0xe8, 0x62, 0x47, 0xbe, 0xa3, 0xd1, 0x9c,
  0x47, 0x31, 0x06, 0x16, 0x6b, 0x3b, 0xf8, 0xad, 0xb0, 0x4c, 0xfb, 0x2b, 0x6b, 0x8e, 0x88, 0x53,
  0x70, 0x75, 0x40, 0x30, 0xaf, 0xfe, 0xc5, 0xf3, 0x0f, 0x86, 0x8f, 0x58, 0x56, 0x67, 0xbd, 0x15,
  0x1f, 0x8a, 0x5c, 0xa3, 0x8a, 0x9a, 0x88, 0xe5, 0xf2, 0xd3, 0x7f, 0xac, 0x56, 0x34, 0x21, 0x24,
  0xf7, 0xde, 0x9d, 0x97, 0x9b, 0xe6, 0x20, 0xd6, 0x3d, 0x48, 0x73, 0x29, 0x23, 0xf9, 0x0f, 0xab,
  0x04, 0xe6, 0x1d, 0x70, 0xee, 0xcb, 0x5a, 0xe5, 0x8a, 0xf4, 0xbc, 0x4b, 0xad, 0x18, 0xec, 0x25,
  0x8a, 0xb8, 0x91, 0xf9, 0x53, 0xf4, 0xe4, 0xce, 0xa7, 0x54, 0xf6, 0x83, 0x66, 0x07, 0xc5, 0x51,
  0x48, 0x20, 0x54, 0x5b, 0x5d, 0xc1, 0x80, 0x04, 0x17, 0x89, 0x14, 0x5a, 0x06, 0x34, 0x86, 0xdb,
  0x8b, 0xaf, 0x04, 0x24, 0xd9, 0xbd, 0xa8, 0x27, 0x09, 0x51, 0xfe, 0x6e, 0x47, 0x3c, 0x29, 0x0a,
  0x64, 0x3e, 0x0f, 0x00, 0xf1, 0xd9, 0xdc, 0xe9, 0xd1, 0x3c, 0xcf, 0xa4, 0xa9, 0x4f, 0x54, 0x02,
  0xcf, 0xa1, 0x40, 0x49, 0x61, 0x2a, 0x07, 0xa2, 0x44, 0x35, 0x20, 0x20, 0x56, 0x1c, 0x27, 0xa2,
  0xff, 0x07, 0x3e, 0x8b, 0xfe, 0x60, 0x42, 0x75, 0x0c, 0xe3, 0x8f, 0xee, 0x33, 0x8d, 0xfc, 0x3b,
  0x53, 0x36, 0x39, 0x10, 0x1a, 0xe3, 0xd2, 0x09, 0x7b, 0x00, 0xd0, 0xbd, 0x39, 0xf4, 0xcc, 0xe3,
  0x42, 0xd4, 0xa5, 0x2d, 0x0b, 0x58, 0xdf, 0x24, 0x7f, 0x85, 0x0a, 0xf2, 0x17, 0x76, 0xdd,
};

UNIT_TEST_STATUS
EFIAPI
TestVerifyX509 (
  IN UNIT_TEST_CONTEXT  Context
  )
{
  BOOLEAN      Status;
  CONST UINT8  *LeafCert;
  UINTN        LeafCertLen;
  UINTN        CertVersion;
  UINT8        Asn1Buffer[1024];
  UINTN        Asn1BufferLen;
  UINTN        SubjectSize;
  UINT8        *Subject;
  UINT8        EndEertFrom[64];
  UINTN        EndEertFromLen;
  UINT8        EndEertTo[64];
  UINTN        EndEertToLen;
  UINT8        DateTime1[64];
  UINT8        DateTime2[64];

  //
  // X509 Certificate Verification.
  //
  Status = X509VerifyCert (mTestCert, sizeof (mTestCert), mTestCaCert, sizeof (mTestCaCert));
  UT_ASSERT_TRUE (Status);

  //
  // X509 Certificate Chain Verification.
  //
  Status = X509VerifyCertChain (mTestCaCert, sizeof (mTestCaCert), mTestBundleCert, sizeof (mTestBundleCert));
  UT_ASSERT_TRUE (Status);

  //
  // X509 Get leaf certificate from cert_chain Verificate.
  //
  Status = X509GetCertFromCertChain (mTestBundleCert, sizeof (mTestBundleCert), -1, &LeafCert, &LeafCertLen);
  UT_ASSERT_TRUE (Status);

  UT_ASSERT_EQUAL (LeafCertLen, sizeof (mTestEndCert));
  UT_ASSERT_MEM_EQUAL (LeafCert, mTestEndCert, LeafCertLen);

  //
  // X509 Get leaf certificate from cert_chain Verificate.
  //
  Status = X509GetCertFromCertChain (mTestBundleCert, sizeof (mTestBundleCert), 2, &LeafCert, &LeafCertLen);
  UT_ASSERT_TRUE (Status);

  UT_ASSERT_EQUAL (LeafCertLen, sizeof (mTestEndCert));
  UT_ASSERT_MEM_EQUAL (LeafCert, mTestEndCert, LeafCertLen);

  //
  // X509 Get root certificate from cert_chain Verificate.
  //
  Status = X509GetCertFromCertChain (mTestBundleCert, sizeof (mTestBundleCert), 0, &LeafCert, &LeafCertLen);
  UT_ASSERT_TRUE (Status);

  UT_ASSERT_EQUAL (LeafCertLen, sizeof (mTestCaCert));
  UT_ASSERT_MEM_EQUAL (LeafCert, mTestCaCert, LeafCertLen);

  //
  // Get version from X509 Certificate.
  //
  CertVersion = 0;
  Status      = X509GetVersion (mTestCert, sizeof (mTestCert), &CertVersion);
  UT_ASSERT_TRUE (Status);

  //
  // Get Serial from X509 Certificate.
  //
  Asn1BufferLen = 1024;
  ZeroMem (Asn1Buffer, Asn1BufferLen);
  Status = X509GetSerialNumber (mTestCert, sizeof (mTestCert), Asn1Buffer, &Asn1BufferLen);
  UT_ASSERT_TRUE (Status);

  //
  // X509 Certificate subject Retrieving.
  //
  SubjectSize = 0;
  Status      = X509GetIssuerName (mTestCert, sizeof (mTestCert), NULL, &SubjectSize);
  UT_ASSERT_TRUE (!Status);
  Subject = AllocatePool (SubjectSize);
  Status  = X509GetIssuerName (mTestCert, sizeof (mTestCert), Subject, &SubjectSize);
  UT_ASSERT_TRUE (Status);
  FreePool (Subject);

  //
  // Get X509GetSubjectAltName.
  //
  Asn1BufferLen = 1024;
  ZeroMem (Asn1Buffer, Asn1BufferLen);
  Status = X509GetExtensionData (mTestEndCert, sizeof (mTestEndCert), mOidSubjectAltName, sizeof (mOidSubjectAltName), Asn1Buffer, &Asn1BufferLen);
  UT_ASSERT_TRUE (Status);

  //
  // Get X509 Validity.
  //
  EndEertFromLen = 64;
  EndEertToLen   = 64;
  Status         = X509GetValidity (
                     mTestEndCert,
                     sizeof (mTestEndCert),
                     EndEertFrom,
                     &EndEertFromLen,
                     EndEertTo,
                     &EndEertToLen
                     );
  UT_ASSERT_TRUE (Status);

  Asn1BufferLen = 64;
  Status        = X509FormatDateTime (
                    "19700101000000Z",
                    DateTime1,
                    &Asn1BufferLen
                    );
  UT_ASSERT_TRUE (Status && (Asn1BufferLen != 0));

  Asn1BufferLen = 64;
  Status        = X509FormatDateTime (
                    "19700201000000Z",
                    DateTime2,
                    &Asn1BufferLen
                    );
  UT_ASSERT_TRUE (Status && (Asn1BufferLen != 0));

  UT_ASSERT_TRUE (X509CompareDateTime (DateTime1, DateTime2) < 0);

  return UNIT_TEST_PASSED;
}

TEST_DESC  mX509Test[] = {
  //
  // -----Description--------------------------------------Class----------------------Function---------------------------------Pre---------------------Post---------Context
  //
  { "TestVerifyX509()", "CryptoPkg.BaseCryptLib.Hkdf", TestVerifyX509, NULL, NULL, NULL },
};

UINTN  mX509TestNum = ARRAY_SIZE (mX509Test);
