<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\HttpKernel\Tests;

use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Config\Loader\LoaderInterface;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\DependencyInjection\Extension\ExtensionInterface;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\Filesystem\Exception\IOException;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Bundle\Bundle;
use Symfony\Component\HttpKernel\Bundle\BundleInterface;
use Symfony\Component\HttpKernel\CacheWarmer\WarmableInterface;
use Symfony\Component\HttpKernel\DependencyInjection\ResettableServicePass;
use Symfony\Component\HttpKernel\DependencyInjection\ServicesResetter;
use Symfony\Component\HttpKernel\HttpKernel;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\HttpKernel\Kernel;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\HttpKernel\Tests\Fixtures\KernelWithoutBundles;
use Symfony\Component\HttpKernel\Tests\Fixtures\ResettableService;

class KernelTest extends TestCase
{
    protected function tearDown(): void
    {
        try {
            (new Filesystem())->remove(__DIR__.'/Fixtures/var');
        } catch (IOException $e) {
        }
    }

    public function testConstructor()
    {
        $env = 'test_env';
        $debug = true;
        $kernel = new KernelForTest($env, $debug);

        $this->assertEquals($env, $kernel->getEnvironment());
        $this->assertEquals($debug, $kernel->isDebug());
        $this->assertFalse($kernel->isBooted());
        $this->assertLessThanOrEqual(microtime(true), $kernel->getStartTime());
    }

    public function testEmptyEnv()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage(\sprintf('Invalid environment provided to "%s": the environment cannot be empty.', KernelForTest::class));

        new KernelForTest('', false);
    }

    public function testClone()
    {
        $env = 'test_env';
        $debug = true;
        $kernel = new KernelForTest($env, $debug);

        $clone = clone $kernel;

        $this->assertEquals($env, $clone->getEnvironment());
        $this->assertEquals($debug, $clone->isDebug());
        $this->assertFalse($clone->isBooted());
        $this->assertLessThanOrEqual(microtime(true), $clone->getStartTime());
    }

    public function testClassNameValidityGetter()
    {
        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('The environment "test.env" contains invalid characters, it can only contain characters allowed in PHP class names.');
        // We check the classname that will be generated by using a $env that
        // contains invalid characters.
        $env = 'test.env';
        $kernel = new KernelForTest($env, false, false);

        $kernel->boot();
    }

    public function testInitializeContainerClearsOldContainers()
    {
        $fs = new Filesystem();
        $legacyContainerDir = __DIR__.'/Fixtures/var/cache/custom/ContainerA123456';
        $fs->mkdir($legacyContainerDir);
        touch($legacyContainerDir.'.legacy');

        $kernel = new CustomProjectDirKernel();
        $kernel->boot();

        $containerDir = __DIR__.'/Fixtures/var/cache/custom/'.substr($kernel->getContainer()::class, 0, 16);
        $this->assertTrue(unlink(__DIR__.'/Fixtures/var/cache/custom/Symfony_Component_HttpKernel_Tests_CustomProjectDirKernelCustomDebugContainer.php.meta'));
        $this->assertFileExists($containerDir);
        $this->assertFileDoesNotExist($containerDir.'.legacy');

        $kernel = new CustomProjectDirKernel(function ($container) { $container->register('foo', 'stdClass')->setPublic(true); });
        $kernel->boot();

        $this->assertFileExists($containerDir);
        $this->assertFileExists($containerDir.'.legacy');

        $this->assertFileDoesNotExist($legacyContainerDir);
        $this->assertFileDoesNotExist($legacyContainerDir.'.legacy');
    }

    public function testBootInitializesBundlesAndContainer()
    {
        $kernel = $this->getKernel(['initializeBundles']);
        $kernel->expects($this->once())
            ->method('initializeBundles');

        $kernel->boot();
    }

    public function testBootSetsTheContainerToTheBundles()
    {
        $bundle = $this->createMock(Bundle::class);
        $bundle->expects($this->once())
            ->method('setContainer');

        $kernel = $this->getKernel(['initializeBundles', 'getBundles']);
        $kernel->expects($this->once())
            ->method('getBundles')
            ->willReturn([$bundle]);

        $kernel->boot();
    }

    public function testBootSetsTheBootedFlagToTrue()
    {
        // use test kernel to access isBooted()
        $kernel = new KernelForTest('test', false);
        $kernel->boot();

        $this->assertTrue($kernel->isBooted());
    }

    public function testClassCacheIsNotLoadedByDefault()
    {
        $kernel = $this->getKernel(['initializeBundles', 'doLoadClassCache'], [], false, KernelForTestWithLoadClassCache::class);
        $kernel->expects($this->never())
            ->method('doLoadClassCache');

        $kernel->boot();
    }

    public function testBootKernelSeveralTimesOnlyInitializesBundlesOnce()
    {
        $kernel = $this->getKernel(['initializeBundles']);
        $kernel->expects($this->once())
            ->method('initializeBundles');

        $kernel->boot();
        $kernel->boot();
    }

    public function testShutdownCallsShutdownOnAllBundles()
    {
        $bundle = $this->createMock(Bundle::class);
        $bundle->expects($this->once())
            ->method('shutdown');

        $kernel = new KernelForTest('test', false, true, [$bundle]);

        $kernel->boot();
        $kernel->shutdown();
    }

    public function testShutdownGivesNullContainerToAllBundles()
    {
        $bundle = $this->createMock(Bundle::class);
        $bundle->expects($this->exactly(2))
            ->method('setContainer')
            ->willReturnCallback(function ($container) {
                if (null !== $container) {
                    $this->assertInstanceOf(ContainerInterface::class, $container);
                }
            })
        ;

        $kernel = new KernelForTest('test', false, true, [$bundle]);

        $kernel->boot();
        $kernel->shutdown();
    }

    public function testHandleCallsHandleOnHttpKernel()
    {
        $type = HttpKernelInterface::MAIN_REQUEST;
        $catch = true;
        $request = new Request();

        $httpKernelMock = $this->getMockBuilder(HttpKernel::class)
            ->disableOriginalConstructor()
            ->getMock();
        $httpKernelMock
            ->expects($this->once())
            ->method('handle')
            ->with($request, $type, $catch);

        $kernel = $this->getKernel(['getHttpKernel']);
        $kernel->expects($this->once())
            ->method('getHttpKernel')
            ->willReturn($httpKernelMock);

        $kernel->handle($request, $type, $catch);
    }

    public function testHandleBootsTheKernel()
    {
        $type = HttpKernelInterface::MAIN_REQUEST;
        $catch = true;
        $request = new Request();

        $httpKernelMock = $this->createStub(HttpKernelInterface::class);

        $kernel = $this->getKernel(['getHttpKernel', 'boot']);
        $kernel->expects($this->once())
            ->method('getHttpKernel')
            ->willReturn($httpKernelMock);

        $kernel->expects($this->once())
            ->method('boot');

        $kernel->handle($request, $type, $catch);
    }

    public function testSerialize()
    {
        $env = 'test_env';
        $debug = true;
        $kernel = new KernelForTest($env, $debug);
        $expected = \sprintf('O:48:"%s":2:{s:11:"environment";s:8:"test_env";s:5:"debug";b:1;}', KernelForTest::class);
        $this->assertEquals($expected, serialize($kernel));
    }

    public function testLocateResourceThrowsExceptionWhenNameIsNotValid()
    {
        $this->expectException(\InvalidArgumentException::class);
        (new KernelForTest('test', false))->locateResource('Foo');
    }

    public function testLocateResourceThrowsExceptionWhenNameIsUnsafe()
    {
        $this->expectException(\RuntimeException::class);
        (new KernelForTest('test', false))->locateResource('@FooBundle/../bar');
    }

    public function testLocateResourceThrowsExceptionWhenBundleDoesNotExist()
    {
        $this->expectException(\InvalidArgumentException::class);
        (new KernelForTest('test', false))->locateResource('@FooBundle/config/routing.xml');
    }

    public function testLocateResourceThrowsExceptionWhenResourceDoesNotExist()
    {
        $bundle = $this->createStub(BundleInterface::class);
        $bundle->method('getPath')->willReturn(__DIR__.'/Fixtures/Bundle1Bundle');
        $this->expectException(\InvalidArgumentException::class);
        $kernel = $this->getKernel(['getBundle']);
        $kernel
            ->expects($this->once())
            ->method('getBundle')
            ->willReturn($bundle)
        ;

        $kernel->locateResource('@Bundle1Bundle/config/routing.xml');
    }

    public function testLocateResourceReturnsTheFirstThatMatches()
    {
        $bundle = $this->createStub(BundleInterface::class);
        $bundle->method('getPath')->willReturn(__DIR__.'/Fixtures/Bundle1Bundle');
        $kernel = $this->getKernel(['getBundle']);
        $kernel
            ->expects($this->once())
            ->method('getBundle')
            ->willReturn($bundle)
        ;

        $this->assertEquals(__DIR__.'/Fixtures/Bundle1Bundle/foo.txt', $kernel->locateResource('@Bundle1Bundle/foo.txt'));
    }

    public function testLocateResourceOnDirectories()
    {
        $bundle = $this->createStub(BundleInterface::class);
        $bundle->method('getName')->willReturn('Bundle1Bundle');
        $bundle->method('getPath')->willReturn(__DIR__.'/Fixtures/Bundle1Bundle');
        $kernel = $this->getKernel(['getBundle']);
        $kernel
            ->expects($this->exactly(2))
            ->method('getBundle')
            ->willReturn($bundle)
        ;

        $this->assertEquals(
            __DIR__.'/Fixtures/Bundle1Bundle/Resources/',
            $kernel->locateResource('@Bundle1Bundle/Resources/')
        );
        $this->assertEquals(
            __DIR__.'/Fixtures/Bundle1Bundle/Resources',
            $kernel->locateResource('@Bundle1Bundle/Resources')
        );
    }

    public function testInitializeBundleThrowsExceptionWhenRegisteringTwoBundlesWithTheSameName()
    {
        $this->expectException(\LogicException::class);
        $this->expectExceptionMessage('Trying to register two bundles with the same name "DuplicateName"');
        $fooBundle = $this->createStub(BundleInterface::class);
        $fooBundle->method('getName')->willReturn('DuplicateName');
        $barBundle = $this->createStub(BundleInterface::class);
        $barBundle->method('getName')->willReturn('DuplicateName');

        $kernel = new KernelForTest('test', false, true, [$fooBundle, $barBundle]);
        $kernel->boot();
    }

    public function testTerminateReturnsSilentlyIfKernelIsNotBooted()
    {
        $kernel = $this->getKernel(['getHttpKernel']);
        $kernel->expects($this->never())
            ->method('getHttpKernel');

        $kernel->terminate(Request::create('/'), new Response());
    }

    public function testTerminateDelegatesTerminationOnlyForTerminableInterface()
    {
        // does not implement TerminableInterface
        $httpKernel = new TestKernel();

        $kernel = $this->getKernel(['getHttpKernel']);
        $kernel->expects($this->once())
            ->method('getHttpKernel')
            ->willReturn($httpKernel);

        $kernel->boot();
        $kernel->terminate(Request::create('/'), new Response());

        $this->assertFalse($httpKernel->terminateCalled, 'terminate() is never called if the kernel class does not implement TerminableInterface');

        // implements TerminableInterface
        $httpKernelMock = $this->getMockBuilder(HttpKernel::class)
            ->disableOriginalConstructor()
            ->onlyMethods(['terminate'])
            ->getMock();

        $httpKernelMock
            ->expects($this->once())
            ->method('terminate');

        $kernel = $this->getKernel(['getHttpKernel']);
        $kernel->expects($this->exactly(2))
            ->method('getHttpKernel')
            ->willReturn($httpKernelMock);

        $kernel->boot();
        $kernel->terminate(Request::create('/'), new Response());
    }

    public function testKernelWithoutBundles()
    {
        $kernel = new KernelWithoutBundles('test', true);
        $kernel->boot();

        $this->assertTrue($kernel->getContainer()->getParameter('test_executed'));
    }

    public function testProjectDirExtension()
    {
        $kernel = new CustomProjectDirKernel();
        $kernel->boot();

        $this->assertSame(__DIR__.'/Fixtures', $kernel->getProjectDir());
        $this->assertSame(__DIR__.\DIRECTORY_SEPARATOR.'Fixtures', $kernel->getContainer()->getParameter('kernel.project_dir'));
    }

    public function testKernelReset()
    {
        $this->tearDown();

        $kernel = new CustomProjectDirKernel();
        $kernel->boot();

        $containerClass = $kernel->getContainer()::class;
        $containerFile = (new \ReflectionClass($kernel->getContainer()))->getFileName();
        unlink(__DIR__.'/Fixtures/var/cache/custom/Symfony_Component_HttpKernel_Tests_CustomProjectDirKernelCustomDebugContainer.php.meta');

        $kernel = new CustomProjectDirKernel();
        $kernel->boot();

        $this->assertInstanceOf($containerClass, $kernel->getContainer());
        $this->assertFileExists($containerFile);
        unlink(__DIR__.'/Fixtures/var/cache/custom/Symfony_Component_HttpKernel_Tests_CustomProjectDirKernelCustomDebugContainer.php.meta');

        $kernel = new CustomProjectDirKernel(function ($container) { $container->register('foo', 'stdClass')->setPublic(true); });
        $kernel->boot();

        $this->assertNotInstanceOf($containerClass, $kernel->getContainer());
        $this->assertFileExists($containerFile);
        $this->assertFileExists(\dirname($containerFile).'.legacy');
    }

    public function testKernelExtension()
    {
        $kernel = new class extends CustomProjectDirKernel implements ExtensionInterface {
            public function load(array $configs, ContainerBuilder $container): void
            {
                $container->setParameter('test.extension-registered', true);
            }

            /**
             * @deprecated since Symfony 7.4, to be removed in Symfony 8.0 together with XML support.
             */
            public function getNamespace(): string
            {
                return '';
            }

            /**
             * @deprecated since Symfony 7.4, to be removed in Symfony 8.0 together with XML support.
             */
            public function getXsdValidationBasePath(): string|false
            {
                return false;
            }

            public function getAlias(): string
            {
                return 'test-extension';
            }
        };
        $kernel->boot();

        $this->assertTrue($kernel->getContainer()->getParameter('test.extension-registered'));
    }

    public function testKernelPass()
    {
        $kernel = new PassKernel();
        $kernel->boot();

        $this->assertTrue($kernel->getContainer()->getParameter('test.processed'));
    }

    public function testWarmup()
    {
        $kernel = new CustomProjectDirKernel();
        $kernel->boot();

        $this->assertTrue($kernel->warmedUp);
        $this->assertSame(realpath($kernel->getBuildDir()), $kernel->warmedUpBuildDir);
    }

    public function testServicesResetter()
    {
        $httpKernelMock = $this->getMockBuilder(HttpKernelInterface::class)
            ->disableOriginalConstructor()
            ->getMock();
        $httpKernelMock
            ->expects($this->exactly(2))
            ->method('handle');

        $kernel = new CustomProjectDirKernel(function ($container) {
            $container->addCompilerPass(new ResettableServicePass());
            $container->register('one', ResettableService::class)
                ->setPublic(true)
                ->addTag('kernel.reset', ['method' => 'reset']);
            $container->register('services_resetter', ServicesResetter::class)->setPublic(true);
        }, $httpKernelMock, 'resetting');

        ResettableService::$counter = 0;

        $request = new Request();

        $kernel->handle($request);
        $kernel->getContainer()->get('one');

        $this->assertEquals(0, ResettableService::$counter);
        $this->assertFalse($kernel->getContainer()->initialized('services_resetter'));

        $kernel->handle($request);

        $this->assertEquals(1, ResettableService::$counter);
    }

    public function testServicesAreNotResetBetweenHttpCacheFragments()
    {
        ResettableService::$counter = 0;
        $fragmentKernel = new FragmentHandlingKernel();

        $kernel = new CustomProjectDirKernel(function (ContainerBuilder $container) {
            $container->addCompilerPass(new ResettableServicePass());
            $container->register('kernel', CustomProjectDirKernel::class)
                ->setSynthetic(true)
                ->setPublic(true);
            $container->register('one', ResettableService::class)
                ->setPublic(true)
                ->addTag('kernel.reset', ['method' => 'reset']);
            $container->register('services_resetter', ServicesResetter::class)->setPublic(true);
            $container->register('http_cache', FragmentRenderingHttpCache::class)
                ->setPublic(true)
                ->addArgument(new Reference('kernel'));
        }, $fragmentKernel, 'http_cache_fragments');

        $kernel->handle(new Request());

        $this->assertSame([
            ['/first-fragment', HttpKernelInterface::MAIN_REQUEST],
            ['/second-fragment', HttpKernelInterface::MAIN_REQUEST],
        ], $fragmentKernel->handledPaths);
        $this->assertSame([0, 0], $fragmentKernel->resetCounters);
        $this->assertSame(0, ResettableService::$counter);

        $kernel->boot();

        $this->assertSame(1, ResettableService::$counter);
    }

    public function testHttpCacheHandlesRequestsAfterKernelBoot()
    {
        $kernel = new CustomProjectDirKernel(static function (ContainerBuilder $container) {
            $container->register('http_cache', RecordingHttpCache::class)
                ->setPublic(true);
        }, new ThrowingHttpKernel(), 'http_cache_worker');

        $kernel->boot();

        $firstResponse = $kernel->handle(Request::create('/worker-first'));
        $secondResponse = $kernel->handle(Request::create('/worker-second'));

        /** @var RecordingHttpCache $httpCache */
        $httpCache = $kernel->getContainer()->get('http_cache');

        $this->assertSame([
            ['/worker-first', HttpKernelInterface::MAIN_REQUEST],
            ['/worker-second', HttpKernelInterface::MAIN_REQUEST],
        ], $httpCache->handledPaths);
        $this->assertSame('cached: /worker-first', $firstResponse->getContent());
        $this->assertSame('cached: /worker-second', $secondResponse->getContent());
    }

    #[Group('time-sensitive')]
    public function testKernelStartTimeIsResetWhileBootingAlreadyBootedKernel()
    {
        $kernel = new KernelForTest('test', true);
        $kernel->boot();
        $preReBoot = $kernel->getStartTime();

        sleep(3600); // Intentionally large value to detect if ClockMock ever breaks
        $kernel->reboot(null);

        $this->assertGreaterThan($preReBoot, $kernel->getStartTime());
    }

    public function testAnonymousKernelGeneratesValidContainerClass()
    {
        $kernel = new class('test', true) extends Kernel {
            public function registerBundles(): iterable
            {
                return [];
            }

            public function registerContainerConfiguration(LoaderInterface $loader): void
            {
            }

            public function getContainerClass(): string
            {
                return parent::getContainerClass();
            }
        };

        $this->assertMatchesRegularExpression('/^[a-zA-Z_\x80-\xff][a-zA-Z0-9_\x80-\xff]*TestDebugContainer$/', $kernel->getContainerClass());
    }

    public function testTrustedParameters()
    {
        $kernel = new CustomProjectDirKernel(function (ContainerBuilder $container) {
            $container->setParameter('kernel.trusted_hosts', '^a{2,3}.com$, ^b{2,}.com$');
            $container->setParameter('kernel.trusted_proxies', 'a,b');
            $container->setParameter('kernel.trusted_headers', 'x-forwarded-for');
        });
        $kernel->boot();

        try {
            $this->assertSame(['{^a{2,3}.com$}i', '{^b{2,}.com$}i'], Request::getTrustedHosts());
            $this->assertSame(['a', 'b'], Request::getTrustedProxies());
            $this->assertSame(Request::HEADER_X_FORWARDED_FOR, Request::getTrustedHeaderSet());
        } finally {
            Request::setTrustedHosts([]);
            Request::setTrustedProxies([], 0);
        }
    }

    /**
     * Returns a mock for the abstract kernel.
     *
     * @param array $methods Additional methods to mock (besides the abstract ones)
     * @param array $bundles Bundles to register
     */
    protected function getKernel(array $methods = [], array $bundles = [], bool $debug = false, string $kernelClass = KernelForTest::class): Kernel
    {
        $methods[] = 'registerBundles';

        $kernelMockBuilder = $this
            ->getMockBuilder($kernelClass)
            ->onlyMethods($methods)
            ->setConstructorArgs(['test', $debug])
        ;

        $kernel = $kernelMockBuilder->getMock();
        $kernel->expects($this->any())
            ->method('registerBundles')
            ->willReturn($bundles)
        ;

        return $kernel;
    }
}

class TestKernel implements HttpKernelInterface
{
    public bool $terminateCalled = false;

    public function terminate(): void
    {
        $this->terminateCalled = true;
    }

    public function handle(Request $request, int $type = self::MAIN_REQUEST, bool $catch = true): Response
    {
    }

    public function getProjectDir(): string
    {
        return __DIR__.'/Fixtures';
    }
}

class CustomProjectDirKernel extends Kernel implements WarmableInterface
{
    public bool $warmedUp = false;

    public ?string $warmedUpBuildDir = null;

    public function __construct(
        private readonly ?\Closure $buildContainer = null,
        private readonly ?HttpKernelInterface $httpKernel = null,
        $env = 'custom',
    ) {
        parent::__construct($env, true);
    }

    public function registerBundles(): iterable
    {
        return [];
    }

    public function registerContainerConfiguration(LoaderInterface $loader): void
    {
    }

    public function getProjectDir(): string
    {
        return __DIR__.'/Fixtures';
    }

    public function warmUp(string $cacheDir, ?string $buildDir = null): array
    {
        $this->warmedUp = true;
        $this->warmedUpBuildDir = $buildDir;

        return [];
    }

    protected function build(ContainerBuilder $container): void
    {
        if ($build = $this->buildContainer) {
            $build($container);
        }
    }

    protected function getHttpKernel(): HttpKernelInterface
    {
        return $this->httpKernel;
    }
}

class PassKernel extends CustomProjectDirKernel implements CompilerPassInterface
{
    public function __construct()
    {
        parent::__construct();
        Kernel::__construct('pass', true);
    }

    public function process(ContainerBuilder $container): void
    {
        $container->setParameter('test.processed', true);
    }
}

class FragmentHandlingKernel implements HttpKernelInterface
{
    public array $handledPaths = [];
    public array $resetCounters = [];

    public function handle(Request $request, int $type = self::MAIN_REQUEST, bool $catch = true): Response
    {
        $this->handledPaths[] = [$request->getPathInfo(), $type];
        $this->resetCounters[] = ResettableService::$counter;

        return new Response($request->getPathInfo());
    }
}

class FragmentRenderingHttpCache implements HttpKernelInterface
{
    public function __construct(
        private KernelInterface $kernel,
        private string $trackedServiceId = 'one',
    ) {
    }

    public function handle(Request $request, int $type = self::MAIN_REQUEST, bool $catch = true): Response
    {
        $this->kernel->boot();
        $this->kernel->getContainer()->get($this->trackedServiceId);

        $responses = [];
        foreach (['/first-fragment', '/second-fragment'] as $path) {
            $responses[] = $this->kernel->handle(Request::create($path), self::MAIN_REQUEST, $catch);
        }

        return new Response(implode('', array_map(static fn (Response $response) => $response->getContent(), $responses)));
    }
}

class RecordingHttpCache implements HttpKernelInterface
{
    public array $handledPaths = [];

    public function handle(Request $request, int $type = self::MAIN_REQUEST, bool $catch = true): Response
    {
        $this->handledPaths[] = [$request->getPathInfo(), $type];

        return new Response('cached: '.$request->getPathInfo());
    }
}

class ThrowingHttpKernel implements HttpKernelInterface
{
    public function handle(Request $request, int $type = self::MAIN_REQUEST, bool $catch = true): Response
    {
        throw new \LogicException('The worker HTTP kernel should not be reached when the http_cache service handles the request.');
    }
}

class KernelForTest extends Kernel
{
    public function __construct(
        string $environment,
        bool $debug,
        private readonly bool $fakeContainer = true,
        private array $registeredBundles = [],
    ) {
        parent::__construct($environment, $debug);
    }

    public function getBundleMap(): array
    {
        return [];
    }

    public function registerBundles(): iterable
    {
        return $this->registeredBundles;
    }

    public function registerContainerConfiguration(LoaderInterface $loader): void
    {
    }

    public function isBooted(): bool
    {
        return $this->booted;
    }

    public function getProjectDir(): string
    {
        return __DIR__;
    }

    protected function initializeContainer(): void
    {
        if ($this->fakeContainer) {
            $this->container = new ContainerBuilder();
        } else {
            parent::initializeContainer();
        }
    }
}

class KernelForTestWithLoadClassCache extends KernelForTest
{
    public function doLoadClassCache(): void
    {
    }
}
