/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import base = require( './../../base' );
import dispatch = require( './../../dispatch' );
import dtypes = require( './../../dtypes' );

/**
* Interface describing the `strided` namespace.
*/
interface Namespace {
	/**
	* Base strided.
	*/
	base: typeof base;

	/**
	* Returns a strided array function interface which performs multiple dispatch and supports alternative indexing semantics.
	*
	* @param fcns - list of strided array functions
	* @param types - one-dimensional list of strided array argument data types
	* @param data - strided array function data (e.g., callbacks)
	* @param nargs - total number of strided array function interface arguments (including strides and offsets)
	* @param nin - number of input strided arrays
	* @param nout - number of output strided arrays
	* @throws first argument must be either a function or an array of functions
	* @throws second argument must be an array of strings
	* @throws third argument must be an array-like object or `null`
	* @throws third and first arguments must have the same number of elements
	* @throws fourth argument must be a positive integer
	* @throws fifth argument must be a nonnegative integer
	* @throws sixth argument must be a nonnegative integer
	* @throws fourth argument must be compatible with the specified number of input and output arrays
	* @throws number of types must match the number of functions times the total number of array arguments for each function
	* @throws interface must accept at least one strided input and/or output array
	* @returns strided array function interface
	*
	* @example
	* var unary = require( `@stdlib/strided/base/unary` ).ndarray;
	* var abs = require( `@stdlib/math/base/special/abs` );
	* var Float64Array = require( `@stdlib/array/float64` );
	*
	* var types = [
	*     'float64', 'float64'
	* ];
	*
	* var data = [
	*     abs
	* ];
	*
	* var strided = ns.dispatch( unary, types, data, 7, 1, 1 );
	*
	* // ...
	*
	* var x = new Float64Array( [ -1.0, -2.0, -3.0, -4.0, -5.0 ] );
	* var y = new Float64Array( [ 0.0, 0.0, 0.0, 0.0, 0.0 ] );
	*
	* strided( x.length, x, 1, 0, y, 1, 0 );
	* // y => <Float64Array>[ 1.0, 2.0, 3.0, 4.0, 5.0 ]
	*/
	dispatch: typeof dispatch;

	/**
	* Returns a list of strided array data types.
	*
	* @returns list of strided array data types
	*
	* @example
	* var list = ns.dtypes();
	* // returns [...]
	*/
	dtypes: typeof dtypes;
}

/**
* Strided.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
